// Image Effects Tool
function initializeImageEffects() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="effects-container">
            <h2>✨ Image Effects</h2>
            <p class="tool-description">Apply stunning effects and adjustments to your images</p>

            <div class="upload-zone" id="effects-upload-zone">
                <div class="upload-icon">🎨</div>
                <h3>Drop images here or click to browse</h3>
                <p>Supports JPG, PNG, WebP, GIF</p>
                <input type="file" id="effects-file-input" accept="image/*" multiple style="display: none;">
            </div>

            <div class="editor-section" id="effects-editor-section" style="display: none;">
                <div class="editor-layout">
                    <div class="preview-panel">
                        <h3>Preview</h3>
                        <div class="image-preview" id="effects-preview">
                            <img id="effects-preview-img" src="" alt="Preview">
                        </div>
                        <div class="preview-controls">
                            <button id="effects-prev-btn" class="btn btn-secondary">← Previous</button>
                            <span id="effects-current-index">1 / 1</span>
                            <button id="effects-next-btn" class="btn btn-secondary">Next →</button>
                        </div>
                    </div>

                    <div class="controls-panel">
                        <h3>Effects & Adjustments</h3>
                        
                        <div class="preset-effects">
                            <h4>Preset Effects</h4>
                            <div class="effect-buttons">
                                <button class="effect-btn" data-effect="none">None</button>
                                <button class="effect-btn" data-effect="vintage">Vintage</button>
                                <button class="effect-btn" data-effect="cool">Cool</button>
                                <button class="effect-btn" data-effect="warm">Warm</button>
                                <button class="effect-btn" data-effect="dramatic">Dramatic</button>
                                <button class="effect-btn" data-effect="fade">Fade</button>
                                <button class="effect-btn" data-effect="vivid">Vivid</button>
                            </div>
                        </div>

                        <div class="adjustment-controls">
                            <h4>Adjustments</h4>
                            
                            <div class="control-group">
                                <label>Brightness: <span id="brightness-value">1.0</span></label>
                                <input type="range" id="brightness-slider" min="0.5" max="2" step="0.1" value="1">
                            </div>

                            <div class="control-group">
                                <label>Contrast: <span id="contrast-value">1.0</span></label>
                                <input type="range" id="contrast-slider" min="0.5" max="2" step="0.1" value="1">
                            </div>

                            <div class="control-group">
                                <label>Saturation: <span id="saturation-value">1.0</span></label>
                                <input type="range" id="saturation-slider" min="0" max="2" step="0.1" value="1">
                            </div>

                            <div class="control-group">
                                <label>Hue Rotate: <span id="hue-value">0°</span></label>
                                <input type="range" id="hue-slider" min="-180" max="180" step="10" value="0">
                            </div>

                            <div class="control-group">
                                <label>
                                    <input type="checkbox" id="negative-check">
                                    Negative (Invert Colors)
                                </label>
                            </div>

                            <div class="control-group">
                                <label>Flip:</label>
                                <select id="flip-select">
                                    <option value="none">None</option>
                                    <option value="horizontal">Horizontal</option>
                                    <option value="vertical">Vertical</option>
                                    <option value="both">Both</option>
                                </select>
                            </div>

                            <div class="control-group">
                                <label>Tint Color:</label>
                                <input type="color" id="tint-color" value="#ffffff">
                                <label>
                                    <input type="checkbox" id="tint-enable">
                                    Enable Tint
                                </label>
                            </div>
                        </div>

                        <div class="action-buttons">
                            <button id="effects-reset-btn" class="btn btn-secondary">Reset All</button>
                            <button id="effects-apply-btn" class="btn btn-primary">Apply Effects</button>
                        </div>
                    </div>
                </div>
            </div>

            <div class="results-section" id="effects-results-section" style="display: none;">
                <h3>Processed Images</h3>
                <div class="results-grid" id="effects-results-grid"></div>
                <button id="effects-new-btn" class="btn btn-secondary">Process New Images</button>
            </div>

            <div class="progress-container" id="effects-progress-container" style="display: none;">
                <div class="progress-bar">
                    <div class="progress-fill" id="effects-progress-fill"></div>
                </div>
                <p class="progress-text" id="effects-progress-text">Processing...</p>
            </div>
        </div>
    `;

    let selectedFiles = [];
    let currentIndex = 0;
    let currentSettings = {
        effect: 'none',
        brightness: 1,
        contrast: 1,
        saturation: 1,
        hue: 0,
        negative: false,
        flip: 'none',
        tint: null
    };

    // Elements
    const uploadZone = document.getElementById('effects-upload-zone');
    const fileInput = document.getElementById('effects-file-input');
    const editorSection = document.getElementById('effects-editor-section');
    const previewImg = document.getElementById('effects-preview-img');
    const currentIndexSpan = document.getElementById('effects-current-index');
    const prevBtn = document.getElementById('effects-prev-btn');
    const nextBtn = document.getElementById('effects-next-btn');
    const effectButtons = document.querySelectorAll('.effect-btn');
    const brightnessSlider = document.getElementById('brightness-slider');
    const contrastSlider = document.getElementById('contrast-slider');
    const saturationSlider = document.getElementById('saturation-slider');
    const hueSlider = document.getElementById('hue-slider');
    const negativeCheck = document.getElementById('negative-check');
    const flipSelect = document.getElementById('flip-select');
    const tintColor = document.getElementById('tint-color');
    const tintEnable = document.getElementById('tint-enable');
    const resetBtn = document.getElementById('effects-reset-btn');
    const applyBtn = document.getElementById('effects-apply-btn');
    const resultsSection = document.getElementById('effects-results-section');
    const resultsGrid = document.getElementById('effects-results-grid');
    const newBtn = document.getElementById('effects-new-btn');
    const progressContainer = document.getElementById('effects-progress-container');
    const progressFill = document.getElementById('effects-progress-fill');
    const progressText = document.getElementById('effects-progress-text');

    // Upload zone click
    uploadZone.addEventListener('click', () => fileInput.click());

    // File input change
    fileInput.addEventListener('change', (e) => handleFiles(e.target.files));

    // Drag and drop
    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#007bff';
        uploadZone.style.background = 'rgba(0, 123, 255, 0.05)';
    });

    uploadZone.addEventListener('dragleave', () => {
        uploadZone.style.borderColor = '#ddd';
        uploadZone.style.background = 'transparent';
    });

    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#ddd';
        uploadZone.style.background = 'transparent';
        handleFiles(e.dataTransfer.files);
    });

    function handleFiles(files) {
        const imageFiles = Array.from(files).filter(file => file.type.startsWith('image/'));
        
        if (imageFiles.length === 0) {
            showNotification('Please select valid image files', 'error');
            return;
        }

        selectedFiles = imageFiles;
        currentIndex = 0;
        displayCurrentImage();
        uploadZone.style.display = 'none';
        editorSection.style.display = 'block';
        resultsSection.style.display = 'none';
    }

    function displayCurrentImage() {
        if (selectedFiles.length === 0) return;

        const reader = new FileReader();
        reader.onload = (e) => {
            previewImg.src = e.target.result;
            updatePreviewWithFilters();
        };
        reader.readAsDataURL(selectedFiles[currentIndex]);

        currentIndexSpan.textContent = `${currentIndex + 1} / ${selectedFiles.length}`;
        prevBtn.disabled = currentIndex === 0;
        nextBtn.disabled = currentIndex === selectedFiles.length - 1;
    }

    // Navigation
    prevBtn.addEventListener('click', () => {
        if (currentIndex > 0) {
            currentIndex--;
            displayCurrentImage();
        }
    });

    nextBtn.addEventListener('click', () => {
        if (currentIndex < selectedFiles.length - 1) {
            currentIndex++;
            displayCurrentImage();
        }
    });

    // Preset effects
    effectButtons.forEach(btn => {
        btn.addEventListener('click', () => {
            effectButtons.forEach(b => b.classList.remove('active'));
            btn.classList.add('active');
            currentSettings.effect = btn.dataset.effect;
            updatePreviewWithFilters();
        });
    });

    // Sliders
    brightnessSlider.addEventListener('input', (e) => {
        currentSettings.brightness = parseFloat(e.target.value);
        document.getElementById('brightness-value').textContent = e.target.value;
        updatePreviewWithFilters();
    });

    contrastSlider.addEventListener('input', (e) => {
        currentSettings.contrast = parseFloat(e.target.value);
        document.getElementById('contrast-value').textContent = e.target.value;
        updatePreviewWithFilters();
    });

    saturationSlider.addEventListener('input', (e) => {
        currentSettings.saturation = parseFloat(e.target.value);
        document.getElementById('saturation-value').textContent = e.target.value;
        updatePreviewWithFilters();
    });

    hueSlider.addEventListener('input', (e) => {
        currentSettings.hue = parseInt(e.target.value);
        document.getElementById('hue-value').textContent = e.target.value + '°';
        updatePreviewWithFilters();
    });

    negativeCheck.addEventListener('change', (e) => {
        currentSettings.negative = e.target.checked;
        updatePreviewWithFilters();
    });

    flipSelect.addEventListener('change', (e) => {
        currentSettings.flip = e.target.value;
        updatePreviewWithFilters();
    });

    tintEnable.addEventListener('change', () => {
        updatePreviewWithFilters();
    });

    tintColor.addEventListener('input', () => {
        if (tintEnable.checked) {
            updatePreviewWithFilters();
        }
    });

    function updatePreviewWithFilters() {
        let filters = [];
        
        if (currentSettings.brightness !== 1) {
            filters.push(`brightness(${currentSettings.brightness})`);
        }
        if (currentSettings.contrast !== 1) {
            filters.push(`contrast(${currentSettings.contrast})`);
        }
        if (currentSettings.saturation !== 1) {
            filters.push(`saturate(${currentSettings.saturation})`);
        }
        if (currentSettings.hue !== 0) {
            filters.push(`hue-rotate(${currentSettings.hue}deg)`);
        }
        if (currentSettings.negative) {
            filters.push('invert(1)');
        }

        previewImg.style.filter = filters.join(' ');

        let transform = [];
        if (currentSettings.flip === 'horizontal') {
            transform.push('scaleX(-1)');
        } else if (currentSettings.flip === 'vertical') {
            transform.push('scaleY(-1)');
        } else if (currentSettings.flip === 'both') {
            transform.push('scale(-1, -1)');
        }
        previewImg.style.transform = transform.join(' ');
    }

    // Reset
    resetBtn.addEventListener('click', () => {
        currentSettings = {
            effect: 'none',
            brightness: 1,
            contrast: 1,
            saturation: 1,
            hue: 0,
            negative: false,
            flip: 'none',
            tint: null
        };

        brightnessSlider.value = 1;
        contrastSlider.value = 1;
        saturationSlider.value = 1;
        hueSlider.value = 0;
        negativeCheck.checked = false;
        flipSelect.value = 'none';
        tintEnable.checked = false;
        effectButtons.forEach(b => b.classList.remove('active'));
        effectButtons[0].classList.add('active');

        document.getElementById('brightness-value').textContent = '1.0';
        document.getElementById('contrast-value').textContent = '1.0';
        document.getElementById('saturation-value').textContent = '1.0';
        document.getElementById('hue-value').textContent = '0°';

        updatePreviewWithFilters();
    });

    // Apply effects
    applyBtn.addEventListener('click', async () => {
        progressContainer.style.display = 'block';
        applyBtn.disabled = true;

        const formData = new FormData();
        selectedFiles.forEach(file => formData.append('files', file));
        
        // Add settings
        Object.keys(currentSettings).forEach(key => {
            if (key === 'tint' && tintEnable.checked) {
                formData.append('tint', tintColor.value);
            } else {
                formData.append(key, currentSettings[key]);
            }
        });

        try {
            const xhr = new XMLHttpRequest();
            
            xhr.upload.addEventListener('progress', (e) => {
                if (e.lengthComputable) {
                    const percent = (e.loaded / e.total) * 50;
                    progressFill.style.width = percent + '%';
                    progressText.textContent = `Uploading... ${Math.round(percent)}%`;
                }
            });

            xhr.addEventListener('load', () => {
                if (xhr.status === 200) {
                    const response = JSON.parse(xhr.responseText);
                    progressFill.style.width = '100%';
                    progressText.textContent = 'Complete!';
                    
                    displayResults(response.files);
                    
                    setTimeout(() => {
                        progressContainer.style.display = 'none';
                        progressFill.style.width = '0%';
                    }, 1000);
                } else {
                    throw new Error('Processing failed');
                }
            });

            xhr.addEventListener('error', () => {
                showNotification('Failed to apply effects', 'error');
                progressContainer.style.display = 'none';
                applyBtn.disabled = false;
            });

            xhr.open('POST', '/api/image/effects');
            xhr.send(formData);

            let progress = 50;
            const interval = setInterval(() => {
                if (progress < 90) {
                    progress += 5;
                    progressFill.style.width = progress + '%';
                    progressText.textContent = `Processing... ${progress}%`;
                } else {
                    clearInterval(interval);
                }
            }, 200);

        } catch (error) {
            console.error('Error:', error);
            showNotification('Failed to apply effects', 'error');
            progressContainer.style.display = 'none';
            applyBtn.disabled = false;
        }
    });

    function displayResults(files) {
        editorSection.style.display = 'none';
        resultsSection.style.display = 'block';
        resultsGrid.innerHTML = '';

        files.forEach(file => {
            const resultItem = document.createElement('div');
            resultItem.className = 'result-item';
            resultItem.innerHTML = `
                <img src="${file.url}" alt="${file.name}">
                <div class="result-info">
                    <p class="file-name">${file.name}</p>
                    <a href="${file.url}" download="${file.name}" class="btn btn-sm btn-primary">
                        Download
                    </a>
                </div>
            `;
            resultsGrid.appendChild(resultItem);
        });

        applyBtn.disabled = false;
        showNotification('Effects applied successfully!', 'success');
    }

    // New images
    newBtn.addEventListener('click', () => {
        selectedFiles = [];
        currentIndex = 0;
        fileInput.value = '';
        uploadZone.style.display = 'flex';
        editorSection.style.display = 'none';
        resultsSection.style.display = 'none';
        resetBtn.click();
    });

    function showNotification(message, type) {
        console.log(`${type}: ${message}`);
        alert(message);
    }
}


// Expose to window for router
window.initializeImageEffects = initializeImageEffects;
