// Advanced Image Cropper Tool
function initializeImageCropper() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="image-cropper-container">
            <h2>✂️ Smart Image Cropper</h2>
            <p class="tool-description">Crop images with preset ratios or custom dimensions</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="crop-input" accept="image/*" onchange="loadImageForCrop(this)">
                    <span>📁 Choose Image to Crop</span>
                </label>
            </div>

            <div id="crop-editor" class="crop-editor hidden">
                <div class="crop-preview">
                    <canvas id="crop-canvas"></canvas>
                </div>

                <div class="crop-controls">
                    <h3>Crop Settings</h3>
                    
                    <div class="form-group">
                        <label>Aspect Ratio:</label>
                        <select id="crop-ratio" onchange="applyCropRatio()">
                            <option value="free">Free Crop</option>
                            <option value="1:1">Square (1:1)</option>
                            <option value="4:3">Standard (4:3)</option>
                            <option value="16:9">Widescreen (16:9)</option>
                            <option value="3:2">Photo (3:2)</option>
                            <option value="9:16">Story (9:16)</option>
                            <option value="custom">Custom</option>
                        </select>
                    </div>

                    <div id="custom-ratio" class="hidden">
                        <div class="form-row">
                            <div class="form-group">
                                <label>Width Ratio:</label>
                                <input type="number" id="ratio-width" value="16" min="1">
                            </div>
                            <div class="form-group">
                                <label>Height Ratio:</label>
                                <input type="number" id="ratio-height" value="9" min="1">
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label>Position:</label>
                        <div class="position-grid">
                            <button class="pos-btn" onclick="setCropPosition('tl')">↖️</button>
                            <button class="pos-btn" onclick="setCropPosition('tc')">⬆️</button>
                            <button class="pos-btn" onclick="setCropPosition('tr')">↗️</button>
                            <button class="pos-btn" onclick="setCropPosition('ml')">⬅️</button>
                            <button class="pos-btn" onclick="setCropPosition('mc')">⏺️</button>
                            <button class="pos-btn" onclick="setCropPosition('mr')">➡️</button>
                            <button class="pos-btn" onclick="setCropPosition('bl')">↙️</button>
                            <button class="pos-btn" onclick="setCropPosition('bc')">⬇️</button>
                            <button class="pos-btn" onclick="setCropPosition('br')">↘️</button>
                        </div>
                    </div>

                    <div class="form-group">
                        <label>Output Format:</label>
                        <select id="crop-format">
                            <option value="png">PNG</option>
                            <option value="jpg">JPG</option>
                            <option value="webp">WebP</option>
                        </select>
                    </div>

                    <button class="btn btn-primary" onclick="cropImage()">Crop Image</button>
                    <button class="btn btn-secondary" onclick="resetCrop()">Reset</button>
                </div>
            </div>

            <div id="crop-result" class="result-section hidden">
                <h3>Cropped Image</h3>
                <img id="cropped-preview" src="" alt="Cropped">
                <button class="btn btn-secondary" onclick="downloadCropped()">💾 Download</button>
            </div>
        </div>
    `;
}

let cropSourceImage = null;
let cropCanvas = null;
let cropCtx = null;
let cropData = { x: 0, y: 0, width: 0, height: 0 };

function loadImageForCrop(input) {
    const file = input.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        cropSourceImage = new Image();
        cropSourceImage.onload = function() {
            initializeCropCanvas();
        };
        cropSourceImage.src = e.target.result;
    };
    reader.readAsDataURL(file);
}

function initializeCropCanvas() {
    cropCanvas = document.getElementById('crop-canvas');
    cropCtx = cropCanvas.getContext('2d');
    
    const maxWidth = 600;
    const scale = Math.min(1, maxWidth / cropSourceImage.width);
    
    cropCanvas.width = cropSourceImage.width * scale;
    cropCanvas.height = cropSourceImage.height * scale;
    
    cropCtx.drawImage(cropSourceImage, 0, 0, cropCanvas.width, cropCanvas.height);
    
    cropData = {
        x: 0,
        y: 0,
        width: cropCanvas.width,
        height: cropCanvas.height
    };
    
    document.getElementById('crop-editor').classList.remove('hidden');
}

function applyCropRatio() {
    const ratio = document.getElementById('crop-ratio').value;
    
    if (ratio === 'custom') {
        document.getElementById('custom-ratio').classList.remove('hidden');
    } else {
        document.getElementById('custom-ratio').classList.add('hidden');
    }
}

function setCropPosition(position) {
    // Implement position-based cropping logic
    alert(`Setting crop position: ${position}`);
}

async function cropImage() {
    if (!cropSourceImage) return;

    const canvas = document.createElement('canvas');
    canvas.width = cropData.width;
    canvas.height = cropData.height;
    
    const ctx = canvas.getContext('2d');
    ctx.drawImage(
        cropSourceImage,
        cropData.x, cropData.y, cropData.width, cropData.height,
        0, 0, cropData.width, cropData.height
    );

    const format = document.getElementById('crop-format').value;
    const mimeType = format === 'jpg' ? 'image/jpeg' : `image/${format}`;
    
    canvas.toBlob(blob => {
        const url = URL.createObjectURL(blob);
        document.getElementById('cropped-preview').src = url;
        document.getElementById('crop-result').classList.remove('hidden');
        
        window.croppedImageUrl = url;
        window.croppedImageBlob = blob;
    }, mimeType, 0.95);
}

function downloadCropped() {
    if (!window.croppedImageBlob) return;
    
    const link = document.createElement('a');
    link.href = window.croppedImageUrl;
    link.download = `cropped_${Date.now()}.${document.getElementById('crop-format').value}`;
    link.click();
}

function resetCrop() {
    if (cropSourceImage) {
        initializeCropCanvas();
    }
}


// Expose to window for router
window.initializeImageCropper = initializeImageCropper;
