// Advanced Image Format Converter with Visual Interface

let converterState = {
    files: [],
    selectedFormat: 'png',
    previewUrls: []
};

// Format configurations with icons and descriptions
const imageFormats = {
    jpg: {
        name: 'JPG/JPEG',
        icon: '📷',
        color: '#FF6B6B',
        description: 'Best for photos, smaller file size',
        quality: true,
        transparency: false
    },
    png: {
        name: 'PNG',
        icon: '🖼️',
        color: '#4ECDC4',
        description: 'Best for graphics, supports transparency',
        quality: false,
        transparency: true
    },
    webp: {
        name: 'WebP',
        icon: '🚀',
        color: '#95E1D3',
        description: 'Modern format, best compression',
        quality: true,
        transparency: true
    },
    gif: {
        name: 'GIF',
        icon: '🎬',
        color: '#F38181',
        description: 'Animated images, limited colors',
        quality: false,
        transparency: true
    },
    bmp: {
        name: 'BMP',
        icon: '🎨',
        color: '#AA96DA',
        description: 'Uncompressed, large file size',
        quality: false,
        transparency: false
    },
    tiff: {
        name: 'TIFF',
        icon: '📄',
        color: '#FCBAD3',
        description: 'Professional, high quality',
        quality: true,
        transparency: true
    },
    avif: {
        name: 'AVIF',
        icon: '⚡',
        color: '#A8E6CF',
        description: 'Next-gen format, excellent compression',
        quality: true,
        transparency: true
    }
};

// Initialize Image Converter
function initializeImageConverter() {
    const container = document.getElementById('imageConverter');
    if (!container) return;

    container.innerHTML = `
        <div class="image-converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-palette"></i> Image Format Converter</h2>
                <p>Convert your images to any format with visual preview</p>
            </div>

            <!-- Upload Section -->
            <div class="converter-upload-section">
                <div class="upload-zone" id="converterUploadZone" onclick="document.getElementById('converterFileInput').click()">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Drop images here or click to upload</h3>
                    <p>Supports: JPG, PNG, WebP, GIF, BMP, TIFF, AVIF</p>
                    <input type="file" id="converterFileInput" accept="image/*" multiple style="display: none;">
                </div>
            </div>

            <!-- Preview Section -->
            <div id="converterPreviewSection" class="converter-preview-section hidden">
                <h3><i class="fas fa-images"></i> Your Images (<span id="imageCount">0</span>)</h3>
                <div id="imagePreviewGrid" class="image-preview-grid"></div>
            </div>

            <!-- Format Selection -->
            <div id="converterFormatSection" class="converter-format-section hidden">
                <h3><i class="fas fa-exchange-alt"></i> Convert To</h3>
                <div class="format-grid">
                    ${Object.entries(imageFormats).map(([key, format]) => `
                        <div class="format-card" data-format="${key}" onclick="selectFormat('${key}')">
                            <div class="format-icon" style="background: ${format.color}">${format.icon}</div>
                            <div class="format-name">${format.name}</div>
                            <div class="format-desc">${format.description}</div>
                            <div class="format-features">
                                ${format.quality ? '<span class="feature-badge">Quality Control</span>' : ''}
                                ${format.transparency ? '<span class="feature-badge">Transparency</span>' : ''}
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>

            <!-- Settings Section -->
            <div id="converterSettings" class="converter-settings hidden">
                <div class="settings-card">
                    <h4>Conversion Settings</h4>
                    <div id="qualityControl" class="setting-group hidden">
                        <label>Quality: <span id="qualityValue">90</span>%</label>
                        <input type="range" id="qualitySlider" min="1" max="100" value="90" 
                            oninput="document.getElementById('qualityValue').textContent = this.value">
                    </div>
                    <div class="setting-group">
                        <label>
                            <input type="checkbox" id="resizeOption" onchange="toggleResize()">
                            Resize images
                        </label>
                    </div>
                    <div id="resizeControls" class="setting-group hidden">
                        <div class="resize-inputs">
                            <div>
                                <label>Width (px)</label>
                                <input type="number" id="resizeWidth" placeholder="Auto" min="1">
                            </div>
                            <div>
                                <label>Height (px)</label>
                                <input type="number" id="resizeHeight" placeholder="Auto" min="1">
                            </div>
                        </div>
                        <label>
                            <input type="checkbox" id="maintainAspect" checked>
                            Maintain aspect ratio
                        </label>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div id="converterActions" class="converter-actions hidden">
                <button class="btn-clear" onclick="clearConverter()">
                    <i class="fas fa-trash"></i> Clear All
                </button>
                <button class="btn-convert" onclick="convertImages()">
                    <i class="fas fa-magic"></i> Convert Images
                </button>
            </div>

            <!-- Progress -->
            <div id="converterProgress" class="converter-progress hidden">
                <div class="progress-bar">
                    <div class="progress-fill" id="converterProgressFill"></div>
                </div>
                <p id="converterProgressText">Converting...</p>
            </div>
        </div>
    `;

    setupConverterEvents();
    selectFormat('png'); // Default selection
}

// Setup event listeners
function setupConverterEvents() {
    const fileInput = document.getElementById('converterFileInput');
    const uploadZone = document.getElementById('converterUploadZone');

    fileInput.addEventListener('change', handleConverterFiles);

    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.classList.add('dragover');
    });

    uploadZone.addEventListener('dragleave', () => {
        uploadZone.classList.remove('dragover');
    });

    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.classList.remove('dragover');
        handleConverterFiles({ target: { files: e.dataTransfer.files } });
    });
}

// Handle file selection
function handleConverterFiles(event) {
    const files = Array.from(event.target.files);
    converterState.files = files;

    // Create preview URLs
    converterState.previewUrls = files.map(file => URL.createObjectURL(file));

    displayImagePreviews();
    showConverterSections();
}

// Display image previews
function displayImagePreviews() {
    const grid = document.getElementById('imagePreviewGrid');
    const count = document.getElementById('imageCount');

    count.textContent = converterState.files.length;

    grid.innerHTML = converterState.files.map((file, index) => `
        <div class="preview-item">
            <img src="${converterState.previewUrls[index]}" alt="${file.name}">
            <div class="preview-info">
                <div class="preview-name" title="${file.name}">${file.name}</div>
                <div class="preview-size">${formatFileSize(file.size)}</div>
            </div>
            <button class="preview-remove" onclick="removeImage(${index})">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `).join('');
}

// Show converter sections
function showConverterSections() {
    document.getElementById('converterPreviewSection').classList.remove('hidden');
    document.getElementById('converterFormatSection').classList.remove('hidden');
    document.getElementById('converterSettings').classList.remove('hidden');
    document.getElementById('converterActions').classList.remove('hidden');
}

// Select format
function selectFormat(format) {
    converterState.selectedFormat = format;

    // Update UI
    document.querySelectorAll('.format-card').forEach(card => {
        card.classList.remove('selected');
    });
    document.querySelector(`[data-format="${format}"]`).classList.add('selected');

    // Show/hide quality control
    const qualityControl = document.getElementById('qualityControl');
    if (imageFormats[format].quality) {
        qualityControl.classList.remove('hidden');
    } else {
        qualityControl.classList.add('hidden');
    }
}

// Toggle resize option
function toggleResize() {
    const resizeControls = document.getElementById('resizeControls');
    const checked = document.getElementById('resizeOption').checked;
    if (checked) {
        resizeControls.classList.remove('hidden');
    } else {
        resizeControls.classList.add('hidden');
    }
}

// Remove image
function removeImage(index) {
    URL.revokeObjectURL(converterState.previewUrls[index]);
    converterState.files.splice(index, 1);
    converterState.previewUrls.splice(index, 1);

    if (converterState.files.length === 0) {
        clearConverter();
    } else {
        displayImagePreviews();
    }
}

// Clear converter
function clearConverter() {
    converterState.previewUrls.forEach(url => URL.revokeObjectURL(url));
    converterState.files = [];
    converterState.previewUrls = [];

    document.getElementById('converterPreviewSection').classList.add('hidden');
    document.getElementById('converterFormatSection').classList.add('hidden');
    document.getElementById('converterSettings').classList.add('hidden');
    document.getElementById('converterActions').classList.add('hidden');
    document.getElementById('converterFileInput').value = '';
}

// Convert images
async function convertImages() {
    if (converterState.files.length === 0) {
        alert('Please select images first');
        return;
    }

    const progressSection = document.getElementById('converterProgress');
    const progressFill = document.getElementById('converterProgressFill');
    const progressText = document.getElementById('converterProgressText');

    progressSection.classList.remove('hidden');
    progressFill.style.width = '0%';

    const formData = new FormData();
    converterState.files.forEach(file => {
        formData.append('files', file);
    });

    formData.append('outputFormat', converterState.selectedFormat);

    // Add quality if applicable
    if (imageFormats[converterState.selectedFormat].quality) {
        formData.append('quality', document.getElementById('qualitySlider').value);
    }

    // Add resize options
    if (document.getElementById('resizeOption').checked) {
        const width = document.getElementById('resizeWidth').value;
        const height = document.getElementById('resizeHeight').value;
        const maintainAspect = document.getElementById('maintainAspect').checked;

        if (width) formData.append('width', width);
        if (height) formData.append('height', height);
        formData.append('maintainAspect', maintainAspect);
    }

    try {
        const xhr = new XMLHttpRequest();

        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                const percentComplete = (e.loaded / e.total) * 100;
                progressFill.style.width = percentComplete + '%';
                progressText.textContent = `Uploading... ${Math.round(percentComplete)}%`;
            }
        });

        xhr.addEventListener('load', () => {
            if (xhr.status === 200) {
                const response = JSON.parse(xhr.responseText);
                progressFill.style.width = '100%';
                progressText.textContent = 'Conversion complete! Downloading...';

                // Download files
                if (response.files && response.files.length > 0) {
                    response.files.forEach((file, index) => {
                        setTimeout(() => {
                            downloadFile(file.url, file.name);
                        }, index * 200);
                    });

                    setTimeout(() => {
                        progressSection.classList.add('hidden');
                        alert(`Successfully converted ${response.files.length} image(s)!`);
                    }, response.files.length * 200 + 1000);
                }
            } else {
                throw new Error('Conversion failed');
            }
        });

        xhr.addEventListener('error', () => {
            alert('An error occurred during conversion');
            progressSection.classList.add('hidden');
        });

        xhr.open('POST', '/api/image/convert');
        xhr.send(formData);

    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred: ' + error.message);
        progressSection.classList.add('hidden');
    }
}

// Download file helper
function downloadFile(url, filename) {
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Format file size
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

// Expose to window for router
window.initializeImageConverter = initializeImageConverter;
