// Image Collage Maker

let collageImages = [];

window.initializeImageCollage = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-th-large"></i> Image Collage Maker</h2>
                <p>Create beautiful photo collages</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('collage-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop images here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP (Upload 2-20 images)</p>
                <input type="file" id="collage-input" accept="image/*" multiple style="display: none;">
            </div>

            <div id="collage-preview" class="preview-section hidden">
                <h3>Selected Images (<span id="collage-count">0</span>)</h3>
                <div id="collage-grid" class="image-grid"></div>
            </div>

            <div id="collage-options" class="options-section hidden">
                <h3>Collage Options</h3>
                
                <div class="form-group">
                    <label>Layout:</label>
                    <select id="collage-layout">
                        <option value="grid">Grid</option>
                        <option value="horizontal">Horizontal Strip</option>
                        <option value="vertical">Vertical Strip</option>
                        <option value="mosaic">Mosaic</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Spacing: <span id="collage-spacing-value">10</span>px</label>
                    <input type="range" id="collage-spacing" min="0" max="50" value="10" oninput="document.getElementById('collage-spacing-value').textContent = this.value">
                </div>

                <div class="form-group">
                    <label>Background Color:</label>
                    <input type="color" id="collage-bg-color" value="#ffffff">
                </div>

                <div class="form-group">
                    <label>Output Width (px):</label>
                    <input type="number" id="collage-width" min="400" max="4000" value="1200" class="form-input">
                </div>

                <button onclick="createCollage()" class="btn-primary">
                    <i class="fas fa-magic"></i> Create Collage
                </button>
            </div>

            <div id="collage-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('collage-input').addEventListener('change', handleCollageUpload);
};

function handleCollageUpload(e) {
    collageImages = Array.from(e.target.files);
    if (!collageImages.length) return;

    if (collageImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }

    if (collageImages.length > 20) {
        alert('Maximum 20 images allowed');
        collageImages = collageImages.slice(0, 20);
    }

    document.getElementById('collage-count').textContent = collageImages.length;
    document.getElementById('collage-preview').classList.remove('hidden');
    document.getElementById('collage-options').classList.remove('hidden');

    const grid = document.getElementById('collage-grid');
    grid.innerHTML = '';

    collageImages.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <img src="${URL.createObjectURL(file)}" alt="${file.name}">
            <div class="image-name">${file.name}</div>
        `;
        grid.appendChild(item);
    });
}

async function createCollage() {
    if (collageImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }

    const resultsDiv = document.getElementById('collage-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Creating collage...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    collageImages.forEach(file => formData.append('images', file));
    formData.append('layout', document.getElementById('collage-layout').value);
    formData.append('spacing', document.getElementById('collage-spacing').value);
    formData.append('backgroundColor', document.getElementById('collage-bg-color').value);
    formData.append('width', document.getElementById('collage-width').value);

    try {
        const response = await fetch('/api/creative/create-collage', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Collage Created!</h3>
                    <img src="${data.url}" alt="Collage" style="max-width: 100%;">
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download Collage
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
