// HEIC to JPG Converter

let heicFiles = [];

window.initializeHeicToJpg = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-mobile-alt"></i> HEIC to JPG Converter</h2>
                <p>Convert iPhone HEIC photos to JPG format</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('heic-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop HEIC images here or click to upload</h3>
                <p>Supports: HEIC, HEIF</p>
                <input type="file" id="heic-input" accept=".heic,.heif" multiple style="display: none;">
            </div>

            <div id="heic-preview" class="preview-section hidden">
                <h3>Selected Images (<span id="heic-count">0</span>)</h3>
                <div id="heic-grid" class="image-grid"></div>
            </div>

            <div id="heic-options" class="options-section hidden">
                <h3>Conversion Options</h3>
                <div class="form-group">
                    <label>Quality: <span id="heic-quality-value">90</span>%</label>
                    <input type="range" id="heic-quality" min="60" max="100" value="90" oninput="document.getElementById('heic-quality-value').textContent = this.value">
                </div>
                <button onclick="convertHeicToJpg()" class="btn-primary">
                    <i class="fas fa-sync"></i> Convert to JPG
                </button>
            </div>

            <div id="heic-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('heic-input').addEventListener('change', handleHeicUpload);
};

function handleHeicUpload(e) {
    heicFiles = Array.from(e.target.files);
    if (!heicFiles.length) return;

    document.getElementById('heic-count').textContent = heicFiles.length;
    document.getElementById('heic-preview').classList.remove('hidden');
    document.getElementById('heic-options').classList.remove('hidden');

    const grid = document.getElementById('heic-grid');
    grid.innerHTML = '';

    heicFiles.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <div class="image-name">${file.name}</div>
            <div class="image-size">${(file.size / 1024).toFixed(2)} KB</div>
        `;
        grid.appendChild(item);
    });
}

async function convertHeicToJpg() {
    if (!heicFiles.length) return;

    const resultsDiv = document.getElementById('heic-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Converting HEIC to JPG...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    heicFiles.forEach(file => formData.append('images', file));
    formData.append('format', 'jpg');
    formData.append('quality', document.getElementById('heic-quality').value);

    try {
        const response = await fetch('/api/convert/heic-to-jpg', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayHeicResults(data.images);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displayHeicResults(images) {
    const resultsDiv = document.getElementById('heic-results');
    let html = '<div class="results-success"><h3>✅ Conversion Complete!</h3><div class="results-grid">';

    images.forEach(img => {
        html += `
            <div class="result-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="result-info">
                    <div class="result-name">${img.filename}</div>
                    <div class="result-size">${(img.size / 1024).toFixed(2)} KB</div>
                    <a href="${img.url}" download="${img.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download
                    </a>
                </div>
            </div>
        `;
    });

    html += '</div></div>';
    resultsDiv.innerHTML = html;
}
