// Grayscale Converter

let grayscaleFiles = [];

window.initializeGrayscaleConverter = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-adjust"></i> Grayscale Converter</h2>
                <p>Convert images to black and white</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('grayscale-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop images here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP, GIF</p>
                <input type="file" id="grayscale-input" accept="image/*" multiple style="display: none;">
            </div>

            <div id="grayscale-preview" class="preview-section hidden">
                <h3>Selected Images (<span id="grayscale-count">0</span>)</h3>
                <div id="grayscale-grid" class="image-grid"></div>
            </div>

            <div id="grayscale-options" class="options-section hidden">
                <h3>Output Options</h3>
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="grayscale-format">
                        <option value="jpg">JPG</option>
                        <option value="png">PNG</option>
                        <option value="webp">WebP</option>
                    </select>
                </div>
                <button onclick="applyGrayscale()" class="btn-primary">
                    <i class="fas fa-palette"></i> Convert to Grayscale
                </button>
            </div>

            <div id="grayscale-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('grayscale-input').addEventListener('change', handleGrayscaleUpload);
};

function handleGrayscaleUpload(e) {
    grayscaleFiles = Array.from(e.target.files);
    if (!grayscaleFiles.length) return;

    document.getElementById('grayscale-count').textContent = grayscaleFiles.length;
    document.getElementById('grayscale-preview').classList.remove('hidden');
    document.getElementById('grayscale-options').classList.remove('hidden');

    const grid = document.getElementById('grayscale-grid');
    grid.innerHTML = '';

    grayscaleFiles.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <img src="${URL.createObjectURL(file)}" alt="${file.name}">
            <div class="image-name">${file.name}</div>
        `;
        grid.appendChild(item);
    });
}

async function applyGrayscale() {
    if (!grayscaleFiles.length) return;

    const resultsDiv = document.getElementById('grayscale-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Converting to grayscale...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    grayscaleFiles.forEach(file => formData.append('images', file));
    formData.append('filter', 'grayscale');
    formData.append('format', document.getElementById('grayscale-format').value);

    try {
        const response = await fetch('/api/image/apply-filter', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayGrayscaleResults(data.images);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displayGrayscaleResults(images) {
    const resultsDiv = document.getElementById('grayscale-results');
    let html = '<div class="results-success"><h3>✅ Grayscale Conversion Complete!</h3><div class="results-grid">';

    images.forEach(img => {
        html += `
            <div class="result-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="result-info">
                    <div class="result-name">${img.filename}</div>
                    <a href="${img.url}" download="${img.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download
                    </a>
                </div>
            </div>
        `;
    });

    html += '</div></div>';
    resultsDiv.innerHTML = html;
}
