// Gradient Generator

window.initializeGradientGenerator = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-fill-drip"></i> Gradient Generator</h2>
                <p>Create beautiful CSS gradients</p>
            </div>

            <div class="gradient-editor">
                <div class="gradient-preview" id="gradient-preview"></div>

                <div class="form-group">
                    <label>Gradient Type:</label>
                    <select id="gradient-type" onchange="updateGradient()">
                        <option value="linear">Linear</option>
                        <option value="radial">Radial</option>
                        <option value="conic">Conic</option>
                    </select>
                </div>

                <div id="linear-options">
                    <div class="form-group">
                        <label>Direction: <span id="direction-value">90</span>°</label>
                        <input type="range" id="gradient-direction" min="0" max="360" value="90" oninput="updateGradient(); document.getElementById('direction-value').textContent = this.value">
                    </div>
                </div>

                <h3>Color Stops</h3>
                <div id="color-stops">
                    <div class="color-stop">
                        <input type="color" value="#667eea" onchange="updateGradient()">
                        <input type="number" value="0" min="0" max="100" onchange="updateGradient()"> %
                        <button onclick="removeColorStop(this)" class="btn-small btn-danger">×</button>
                    </div>
                    <div class="color-stop">
                        <input type="color" value="#764ba2" onchange="updateGradient()">
                        <input type="number" value="100" min="0" max="100" onchange="updateGradient()"> %
                        <button onclick="removeColorStop(this)" class="btn-small btn-danger">×</button>
                    </div>
                </div>

                <button onclick="addColorStop()" class="btn-secondary">
                    <i class="fas fa-plus"></i> Add Color Stop
                </button>

                <h3>CSS Code</h3>
                <textarea id="gradient-code" rows="4" readonly></textarea>
                
                <div class="button-group">
                    <button onclick="copyGradientCode()" class="btn-secondary">
                        <i class="fas fa-copy"></i> Copy CSS
                    </button>
                    <button onclick="exportGradientImage()" class="btn-primary">
                        <i class="fas fa-download"></i> Export as Image
                    </button>
                </div>
            </div>
        </div>
    `;

    updateGradient();
};

function updateGradient() {
    const type = document.getElementById('gradient-type').value;
    const direction = document.getElementById('gradient-direction').value;
    const stops = Array.from(document.querySelectorAll('.color-stop')).map(stop => {
        const color = stop.querySelector('input[type="color"]').value;
        const position = stop.querySelector('input[type="number"]').value;
        return `${color} ${position}%`;
    }).join(', ');

    let gradient;
    if (type === 'linear') {
        gradient = `linear-gradient(${direction}deg, ${stops})`;
    } else if (type === 'radial') {
        gradient = `radial-gradient(circle, ${stops})`;
    } else {
        gradient = `conic-gradient(${stops})`;
    }

    document.getElementById('gradient-preview').style.background = gradient;
    document.getElementById('gradient-code').value = `background: ${gradient};`;
}

function addColorStop() {
    const container = document.getElementById('color-stops');
    const newStop = document.createElement('div');
    newStop.className = 'color-stop';
    newStop.innerHTML = `
        <input type="color" value="#${Math.floor(Math.random()*16777215).toString(16)}" onchange="updateGradient()">
        <input type="number" value="50" min="0" max="100" onchange="updateGradient()"> %
        <button onclick="removeColorStop(this)" class="btn-small btn-danger">×</button>
    `;
    container.appendChild(newStop);
    updateGradient();
}

function removeColorStop(btn) {
    if (document.querySelectorAll('.color-stop').length > 2) {
        btn.closest('.color-stop').remove();
        updateGradient();
    } else {
        alert('Gradient needs at least 2 color stops');
    }
}

function copyGradientCode() {
    const code = document.getElementById('gradient-code');
    code.select();
    document.execCommand('copy');
    alert('Gradient CSS copied to clipboard!');
}

function exportGradientImage() {
    const canvas = document.createElement('canvas');
    canvas.width = 800;
    canvas.height = 600;
    const ctx = canvas.getContext('2d');
    
    const preview = document.getElementById('gradient-preview');
    const gradient = window.getComputedStyle(preview).background;
    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    
    const link = document.createElement('a');
    link.download = 'gradient.png';
    link.href = canvas.toDataURL();
    link.click();
}
