// GIF Maker from Images

let gifImages = [];

window.initializeGifMaker = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-film"></i> GIF Maker</h2>
                <p>Create animated GIFs from multiple images</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('gif-images-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop images here or click to upload</h3>
                <p>Select multiple images in the order you want them</p>
                <input type="file" id="gif-images-input" accept="image/*" multiple style="display: none;">
            </div>

            <div id="gif-preview" class="preview-section hidden">
                <h3>Selected Images (<span id="gif-count">0</span>)</h3>
                <div id="gif-grid" class="image-grid sortable"></div>
            </div>

            <div id="gif-options" class="options-section hidden">
                <h3>GIF Options</h3>
                <div class="form-group">
                    <label>Frame Delay: <span id="gif-delay-value">500</span>ms</label>
                    <input type="range" id="gif-delay" min="100" max="2000" step="100" value="500" oninput="document.getElementById('gif-delay-value').textContent = this.value">
                </div>
                <div class="form-group">
                    <label>Loop:</label>
                    <select id="gif-loop">
                        <option value="0">Forever</option>
                        <option value="1">Once</option>
                        <option value="3">3 times</option>
                        <option value="5">5 times</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Quality:</label>
                    <select id="gif-quality">
                        <option value="10">High</option>
                        <option value="20" selected>Medium</option>
                        <option value="30">Low (Smaller file)</option>
                    </select>
                </div>
                <button onclick="createGif()" class="btn-primary">
                    <i class="fas fa-magic"></i> Create GIF
                </button>
            </div>

            <div id="gif-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('gif-images-input').addEventListener('change', handleGifImagesUpload);
};

function handleGifImagesUpload(e) {
    gifImages = Array.from(e.target.files);
    if (gifImages.length < 2) {
        alert('Please select at least 2 images to create a GIF');
        return;
    }

    document.getElementById('gif-count').textContent = gifImages.length;
    document.getElementById('gif-preview').classList.remove('hidden');
    document.getElementById('gif-options').classList.remove('hidden');

    const grid = document.getElementById('gif-grid');
    grid.innerHTML = '';

    gifImages.forEach((file, index) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <div class="image-order">${index + 1}</div>
            <img src="${URL.createObjectURL(file)}" alt="${file.name}">
            <div class="image-name">${file.name}</div>
        `;
        grid.appendChild(item);
    });
}

async function createGif() {
    if (gifImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }

    const resultsDiv = document.getElementById('gif-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Creating GIF animation...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    gifImages.forEach(file => formData.append('images', file));
    formData.append('delay', document.getElementById('gif-delay').value);
    formData.append('loop', document.getElementById('gif-loop').value);
    formData.append('quality', document.getElementById('gif-quality').value);

    try {
        const response = await fetch('/api/creative/create-gif', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayGifResult(data);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displayGifResult(data) {
    const resultsDiv = document.getElementById('gif-results');
    resultsDiv.innerHTML = `
        <div class="results-success">
            <h3>✅ GIF Created Successfully!</h3>
            <div class="gif-preview">
                <img src="${data.url}" alt="Generated GIF">
            </div>
            <div class="file-info">
                <p><strong>Frames:</strong> ${data.frames}</p>
                <p><strong>Size:</strong> ${(data.size / 1024).toFixed(2)} KB</p>
            </div>
            <a href="${data.url}" download="${data.filename}" class="btn-download">
                <i class="fas fa-download"></i> Download GIF
            </a>
        </div>
    `;
}
