// Favicon Generator Tool
function initializeFaviconGenerator() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="favicon-generator-container">
            <h2>🎨 Favicon Generator</h2>
            <p class="tool-description">Generate multi-size favicons from your image</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="favicon-file-input" accept="image/*" onchange="previewFaviconImage(this)">
                    <span>📁 Choose Image (Recommended: 512x512px square)</span>
                </label>
            </div>

            <div id="favicon-preview-section" class="preview-section hidden">
                <h3>Preview</h3>
                <img id="favicon-preview" src="" alt="Preview" style="max-width: 200px; border: 1px solid #ddd; padding: 10px;">
            </div>

            <div id="favicon-options" class="hidden">
                <h3>Generate Options</h3>
                <div class="checkbox-group">
                    <label>
                        <input type="checkbox" checked value="16"> 16x16 (Classic)
                    </label>
                    <label>
                        <input type="checkbox" checked value="32"> 32x32 (Standard)
                    </label>
                    <label>
                        <input type="checkbox" checked value="48"> 48x48 (Modern)
                    </label>
                    <label>
                        <input type="checkbox" checked value="64"> 64x64 (HD)
                    </label>
                    <label>
                        <input type="checkbox" checked value="128"> 128x128 (Retina)
                    </label>
                    <label>
                        <input type="checkbox" checked value="256"> 256x256 (High-DPI)
                    </label>
                </div>

                <button class="btn btn-primary" onclick="generateFavicons()">Generate Favicons</button>
            </div>

            <div id="favicon-results" class="results-section hidden">
                <h3>Generated Favicons</h3>
                <div id="favicon-grid" class="favicon-grid"></div>
                <div class="button-group">
                    <button class="btn btn-secondary" onclick="downloadAllFavicons()">📦 Download All as ZIP</button>
                </div>
                <div class="code-section">
                    <h4>HTML Code</h4>
                    <textarea id="favicon-html-code" rows="6" readonly></textarea>
                    <button class="btn btn-secondary" onclick="copyFaviconCode()">📋 Copy Code</button>
                </div>
            </div>
        </div>
    `;
}

let faviconFile = null;
let generatedFavicons = [];

function previewFaviconImage(input) {
    faviconFile = input.files[0];
    if (!faviconFile) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        document.getElementById('favicon-preview').src = e.target.result;
        document.getElementById('favicon-preview-section').classList.remove('hidden');
        document.getElementById('favicon-options').classList.remove('hidden');
    };
    reader.readAsDataURL(faviconFile);
}

async function generateFavicons() {
    if (!faviconFile) {
        alert('Please select an image first');
        return;
    }

    const checkboxes = document.querySelectorAll('#favicon-options input[type="checkbox"]:checked');
    const sizes = Array.from(checkboxes).map(cb => parseInt(cb.value));

    if (sizes.length === 0) {
        alert('Please select at least one size');
        return;
    }

    const formData = new FormData();
    formData.append('image', faviconFile);
    formData.append('sizes', JSON.stringify(sizes));

    try {
        const response = await fetch('/api/utility/favicon-generate', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            generatedFavicons = data.favicons;
            displayFaviconResults(data.favicons);
        } else {
            alert('Failed to generate favicons: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while generating favicons');
    }
}

function displayFaviconResults(favicons) {
    const grid = document.getElementById('favicon-grid');
    grid.innerHTML = '';

    favicons.forEach(favicon => {
        const item = document.createElement('div');
        item.className = 'favicon-item';
        item.innerHTML = `
            <img src="${favicon.url}" alt="${favicon.size}x${favicon.size}">
            <div class="favicon-info">
                <strong>${favicon.size}x${favicon.size}</strong>
                <button onclick="downloadFavicon('${favicon.url}', '${favicon.filename}')" class="btn-small">Download</button>
            </div>
        `;
        grid.appendChild(item);
    });

    // Generate HTML code
    const htmlCode = favicons.map(f => 
        `<link rel="icon" type="image/png" sizes="${f.size}x${f.size}" href="/favicons/${f.filename}">`
    ).join('\n');
    
    document.getElementById('favicon-html-code').value = htmlCode;
    document.getElementById('favicon-results').classList.remove('hidden');
}

function downloadFavicon(url, filename) {
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    link.click();
}

async function downloadAllFavicons() {
    // Create an array of file URLs for ZIP download
    const files = generatedFavicons.map(f => ({
        url: f.url,
        filename: f.filename
    }));

    try {
        // Download each as data URLs and create ZIP client-side
        const zip = new JSZip();
        
        for (const file of files) {
            const response = await fetch(file.url);
            const blob = await response.blob();
            zip.file(file.filename, blob);
        }

        const zipBlob = await zip.generateAsync({ type: 'blob' });
        const url = URL.createObjectURL(zipBlob);
        
        const link = document.createElement('a');
        link.href = url;
        link.download = 'favicons.zip';
        link.click();
        
        URL.revokeObjectURL(url);
    } catch (error) {
        console.error('Error creating ZIP:', error);
        alert('Failed to create ZIP file. Please download favicons individually.');
    }
}

function copyFaviconCode() {
    const textarea = document.getElementById('favicon-html-code');
    textarea.select();
    document.execCommand('copy');
    
    const btn = event.target;
    const originalText = btn.textContent;
    btn.textContent = '✅ Copied!';
    setTimeout(() => btn.textContent = originalText, 2000);
}

// Expose to window for router
window.initializeFaviconGenerator = initializeFaviconGenerator;
