// Expense Tracker

let expenses = [];

window.initializeExpenseTracker = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-wallet"></i> Expense Tracker</h2>
                <p>Track and manage your expenses</p>
            </div>

            <div class="expense-form">
                <h3>Add Expense</h3>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Description:</label>
                        <input type="text" id="expense-description" class="form-input" placeholder="Coffee, Groceries, etc.">
                    </div>

                    <div class="form-group">
                        <label>Amount:</label>
                        <input type="number" id="expense-amount" class="form-input" step="0.01" placeholder="0.00">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Category:</label>
                        <select id="expense-category">
                            <option value="Food">Food</option>
                            <option value="Transport">Transport</option>
                            <option value="Shopping">Shopping</option>
                            <option value="Bills">Bills</option>
                            <option value="Entertainment">Entertainment</option>
                            <option value="Health">Health</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Date:</label>
                        <input type="date" id="expense-date" class="form-input">
                    </div>
                </div>

                <button onclick="addExpense()" class="btn-primary">
                    <i class="fas fa-plus"></i> Add Expense
                </button>
            </div>

            <div id="expense-summary" class="expense-summary hidden">
                <h3>Summary</h3>
                <div class="summary-cards">
                    <div class="summary-card">
                        <div class="summary-label">Total Expenses</div>
                        <div class="summary-value" id="total-expenses">$0.00</div>
                    </div>
                    <div class="summary-card">
                        <div class="summary-label">Number of Transactions</div>
                        <div class="summary-value" id="total-transactions">0</div>
                    </div>
                    <div class="summary-card">
                        <div class="summary-label">Average per Transaction</div>
                        <div class="summary-value" id="avg-expense">$0.00</div>
                    </div>
                </div>
            </div>

            <div id="expense-list" class="expense-list hidden">
                <div class="list-header">
                    <h3>Expense History</h3>
                    <div class="button-group">
                        <button onclick="clearExpenses()" class="btn-secondary">
                            <i class="fas fa-trash"></i> Clear All
                        </button>
                        <button onclick="exportExpenses()" class="btn-primary">
                            <i class="fas fa-download"></i> Export CSV
                        </button>
                    </div>
                </div>
                <div id="expenses-container"></div>
            </div>
        </div>
    `;

    // Set today's date as default
    document.getElementById('expense-date').value = new Date().toISOString().split('T')[0];
    loadExpenses();
};

function addExpense() {
    const description = document.getElementById('expense-description').value.trim();
    const amount = parseFloat(document.getElementById('expense-amount').value);
    const category = document.getElementById('expense-category').value;
    const date = document.getElementById('expense-date').value;

    if (!description || !amount || amount <= 0) {
        alert('Please enter valid description and amount');
        return;
    }

    const expense = {
        id: Date.now(),
        description,
        amount,
        category,
        date
    };

    expenses.unshift(expense);
    saveExpenses();
    renderExpenses();

    // Clear form
    document.getElementById('expense-description').value = '';
    document.getElementById('expense-amount').value = '';
}

function renderExpenses() {
    if (expenses.length === 0) {
        document.getElementById('expense-summary').classList.add('hidden');
        document.getElementById('expense-list').classList.add('hidden');
        return;
    }

    // Update summary
    const total = expenses.reduce((sum, exp) => sum + exp.amount, 0);
    const count = expenses.length;
    const average = total / count;

    document.getElementById('total-expenses').textContent = `$${total.toFixed(2)}`;
    document.getElementById('total-transactions').textContent = count;
    document.getElementById('avg-expense').textContent = `$${average.toFixed(2)}`;
    document.getElementById('expense-summary').classList.remove('hidden');

    // Render expense list
    const container = document.getElementById('expenses-container');
    container.innerHTML = expenses.map(exp => `
        <div class="expense-item">
            <div class="expense-info">
                <div class="expense-description">${exp.description}</div>
                <div class="expense-meta">
                    <span class="expense-category">${exp.category}</span>
                    <span class="expense-date">${new Date(exp.date).toLocaleDateString()}</span>
                </div>
            </div>
            <div class="expense-actions">
                <div class="expense-amount">$${exp.amount.toFixed(2)}</div>
                <button onclick="deleteExpense(${exp.id})" class="btn-delete">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        </div>
    `).join('');

    document.getElementById('expense-list').classList.remove('hidden');
}

function deleteExpense(id) {
    expenses = expenses.filter(exp => exp.id !== id);
    saveExpenses();
    renderExpenses();
}

function clearExpenses() {
    if (confirm('Are you sure you want to clear all expenses?')) {
        expenses = [];
        saveExpenses();
        renderExpenses();
    }
}

function saveExpenses() {
    localStorage.setItem('expenses', JSON.stringify(expenses));
}

function loadExpenses() {
    const saved = localStorage.getItem('expenses');
    if (saved) {
        expenses = JSON.parse(saved);
        renderExpenses();
    }
}

function exportExpenses() {
    if (expenses.length === 0) {
        alert('No expenses to export');
        return;
    }

    const csv = 'Date,Description,Category,Amount\n' + 
        expenses.map(exp => `${exp.date},"${exp.description}",${exp.category},${exp.amount}`).join('\n');

    const blob = new Blob([csv], { type: 'text/csv' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = `expenses_${new Date().toISOString().split('T')[0]}.csv`;
    link.click();
}
