// Diff Checker

window.initializeDiffChecker = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-code-branch"></i> Text Diff Checker</h2>
                <p>Compare two texts and find differences</p>
            </div>

            <div class="diff-container">
                <div class="diff-panel">
                    <h3>Original Text</h3>
                    <textarea id="diff-text1" rows="15" class="form-input code-input" placeholder="Paste first text here..."></textarea>
                </div>

                <div class="diff-panel">
                    <h3>Modified Text</h3>
                    <textarea id="diff-text2" rows="15" class="form-input code-input" placeholder="Paste second text here..."></textarea>
                </div>
            </div>

            <div class="form-group">
                <label>
                    <input type="checkbox" id="diff-ignore-case"> Ignore case
                </label>
                <label>
                    <input type="checkbox" id="diff-ignore-whitespace"> Ignore whitespace
                </label>
            </div>

            <button onclick="compareDiff()" class="btn-primary">
                <i class="fas fa-exchange-alt"></i> Compare Texts
            </button>

            <div id="diff-results" class="results-section hidden">
                <h3>Differences:</h3>
                <div id="diff-stats" class="stats-bar"></div>
                <div id="diff-output" class="diff-output"></div>
            </div>
        </div>
    `;
};

function compareDiff() {
    let text1 = document.getElementById('diff-text1').value;
    let text2 = document.getElementById('diff-text2').value;

    const ignoreCase = document.getElementById('diff-ignore-case').checked;
    const ignoreWhitespace = document.getElementById('diff-ignore-whitespace').checked;

    if (ignoreCase) {
        text1 = text1.toLowerCase();
        text2 = text2.toLowerCase();
    }

    if (ignoreWhitespace) {
        text1 = text1.replace(/\s+/g, ' ').trim();
        text2 = text2.replace(/\s+/g, ' ').trim();
    }

    const lines1 = text1.split('\n');
    const lines2 = text2.split('\n');

    const diff = calculateDiff(lines1, lines2);
    displayDiff(diff);

    document.getElementById('diff-results').classList.remove('hidden');
}

function calculateDiff(lines1, lines2) {
    const result = [];
    let additions = 0;
    let deletions = 0;
    let unchanged = 0;

    const maxLen = Math.max(lines1.length, lines2.length);

    for (let i = 0; i < maxLen; i++) {
        const line1 = lines1[i];
        const line2 = lines2[i];

        if (line1 === line2) {
            if (line1 !== undefined) {
                result.push({ type: 'unchanged', content: line1 });
                unchanged++;
            }
        } else {
            if (line1 !== undefined) {
                result.push({ type: 'deleted', content: line1 });
                deletions++;
            }
            if (line2 !== undefined) {
                result.push({ type: 'added', content: line2 });
                additions++;
            }
        }
    }

    return { lines: result, additions, deletions, unchanged };
}

function displayDiff(diff) {
    const statsDiv = document.getElementById('diff-stats');
    statsDiv.innerHTML = `
        <span class="stat-item"><strong>Additions:</strong> <span style="color: green;">${diff.additions}</span></span>
        <span class="stat-item"><strong>Deletions:</strong> <span style="color: red;">${diff.deletions}</span></span>
        <span class="stat-item"><strong>Unchanged:</strong> ${diff.unchanged}</span>
    `;

    const outputDiv = document.getElementById('diff-output');
    let html = '';

    diff.lines.forEach(line => {
        let className = '';
        let symbol = '';

        switch(line.type) {
            case 'added':
                className = 'diff-added';
                symbol = '+ ';
                break;
            case 'deleted':
                className = 'diff-deleted';
                symbol = '- ';
                break;
            case 'unchanged':
                className = 'diff-unchanged';
                symbol = '  ';
                break;
        }

        html += `<div class="${className}">${symbol}${escapeHtml(line.content)}</div>`;
    });

    outputDiv.innerHTML = html;
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
