// CSV to JSON Converter

let csvFile = null;

window.initializeCsvToJson = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-table"></i> CSV to JSON Converter</h2>
                <p>Convert CSV files to JSON format</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('csv-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop CSV file here or click to upload</h3>
                <p>Supports: CSV, TSV</p>
                <input type="file" id="csv-input" accept=".csv,.tsv" style="display: none;">
            </div>

            <div id="csv-preview" class="preview-section hidden">
                <h3>File Preview:</h3>
                <div id="csv-content" class="code-preview"></div>
            </div>

            <div id="csv-options" class="options-section hidden">
                <h3>Conversion Options</h3>
                <div class="form-group">
                    <label>Delimiter:</label>
                    <select id="csv-delimiter">
                        <option value=",">Comma (,)</option>
                        <option value=";">Semicolon (;)</option>
                        <option value="\t">Tab</option>
                        <option value="|">Pipe (|)</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" id="csv-has-header" checked> First row is header
                    </label>
                </div>
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="csv-format">
                        <option value="array">Array of Objects</option>
                        <option value="object">Object with Keys</option>
                        <option value="pretty">Pretty Print</option>
                        <option value="minified">Minified</option>
                    </select>
                </div>
                <button onclick="convertCSVtoJSON()" class="btn-primary">
                    <i class="fas fa-sync"></i> Convert to JSON
                </button>
            </div>

            <div id="csv-results" class="results-section hidden">
                <h3>JSON Output:</h3>
                <textarea id="json-result" rows="15" class="form-input code-input" readonly></textarea>
                <div class="button-group">
                    <button onclick="copyJSONResult()" class="btn-download">
                        <i class="fas fa-copy"></i> Copy JSON
                    </button>
                    <button onclick="downloadJSON()" class="btn-download">
                        <i class="fas fa-download"></i> Download JSON
                    </button>
                </div>
            </div>
        </div>
    `;

    document.getElementById('csv-input').addEventListener('change', handleCSVUpload);
};

function handleCSVUpload(e) {
    csvFile = e.target.files[0];
    if (!csvFile) return;

    const reader = new FileReader();
    reader.onload = function(event) {
        const content = event.target.result;
        const preview = content.substring(0, 500) + (content.length > 500 ? '...' : '');
        document.getElementById('csv-content').textContent = preview;
        document.getElementById('csv-preview').classList.remove('hidden');
        document.getElementById('csv-options').classList.remove('hidden');
    };
    reader.readAsText(csvFile);
}

function convertCSVtoJSON() {
    if (!csvFile) return;

    const reader = new FileReader();
    reader.onload = function(event) {
        const csvContent = event.target.result;
        const delimiter = document.getElementById('csv-delimiter').value;
        const hasHeader = document.getElementById('csv-has-header').checked;
        const format = document.getElementById('csv-format').value;

        try {
            const lines = csvContent.split('\n').filter(line => line.trim());
            let headers = [];
            let data = [];

            if (hasHeader) {
                headers = lines[0].split(delimiter).map(h => h.trim());
                data = lines.slice(1);
            } else {
                headers = Array.from({length: lines[0].split(delimiter).length}, (_, i) => `column${i + 1}`);
                data = lines;
            }

            const jsonArray = data.map(line => {
                const values = line.split(delimiter);
                const obj = {};
                headers.forEach((header, index) => {
                    obj[header] = values[index]?.trim() || '';
                });
                return obj;
            });

            let output;
            if (format === 'pretty') {
                output = JSON.stringify(jsonArray, null, 2);
            } else if (format === 'minified') {
                output = JSON.stringify(jsonArray);
            } else if (format === 'array') {
                output = JSON.stringify(jsonArray, null, 2);
            } else {
                const result = { data: jsonArray, count: jsonArray.length };
                output = JSON.stringify(result, null, 2);
            }

            document.getElementById('json-result').value = output;
            document.getElementById('csv-results').classList.remove('hidden');
        } catch (error) {
            alert('Error converting CSV: ' + error.message);
        }
    };
    reader.readAsText(csvFile);
}

function copyJSONResult() {
    const output = document.getElementById('json-result');
    output.select();
    document.execCommand('copy');
    alert('JSON copied to clipboard!');
}

function downloadJSON() {
    const content = document.getElementById('json-result').value;
    const blob = new Blob([content], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = 'converted.json';
    link.click();
    URL.revokeObjectURL(url);
}
