// Crop PDF - Visual Interactive Cropping
let cropPdfFile = null;
let cropPdfDoc = null;
let currentCropPage = 1;
let cropData = { x: 0, y: 0, width: 0, height: 0 };
let isDragging = false;
let dragStart = { x: 0, y: 0 };
let canvasScale = 1;
let pdfPages = [];

window.initializeCropPdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-crop-alt"></i> Crop PDF Pages</h2>
                <p>Draw a rectangle to select the area you want to keep</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="crop-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <p>Upload a PDF to crop</p>
                    <input type="file" id="crop-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="crop-editor" class="hidden" style="margin-top: 2rem;">
                <div style="background: white; padding: 1.5rem; border-radius: 12px; margin-bottom: 1rem;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem; flex-wrap: wrap; gap: 1rem;">
                        <div>
                            <h3 style="margin: 0; margin-bottom: 0.5rem;">✂️ Draw crop area on the page</h3>
                            <p style="margin: 0; color: var(--text-secondary); font-size: 0.9rem;">
                                <i class="fas fa-info-circle"></i> Click and drag to select the area you want to keep
                            </p>
                        </div>
                        <div style="display: flex; gap: 0.5rem;">
                            <button onclick="prevCropPage()" class="btn-secondary" id="prev-page-btn">
                                <i class="fas fa-chevron-left"></i> Previous
                            </button>
                            <span id="crop-page-info" style="padding: 0.5rem 1rem; background: var(--bg-tertiary); border-radius: 6px; font-weight: 500;">
                                Page 1 / 1
                            </span>
                            <button onclick="nextCropPage()" class="btn-secondary" id="next-page-btn">
                                Next <i class="fas fa-chevron-right"></i>
                            </button>
                        </div>
                    </div>

                    <div class="crop-canvas-wrapper" style="position: relative; display: inline-block; margin: 0 auto; max-width: 100%; overflow: auto;">
                        <canvas id="crop-canvas" style="border: 2px solid var(--border-color); border-radius: 8px; cursor: crosshair; display: block; max-width: 100%;"></canvas>
                        <div id="crop-overlay" style="position: absolute; border: 2px solid var(--primary-color); background: rgba(99, 102, 241, 0.2); pointer-events: none; display: none;">
                            <div style="position: absolute; top: -25px; left: 0; background: var(--primary-color); color: white; padding: 2px 8px; border-radius: 4px; font-size: 12px; white-space: nowrap;">
                                Drag to select area
                            </div>
                        </div>
                    </div>

                    <div style="margin-top: 1.5rem; display: flex; gap: 1rem; flex-wrap: wrap; align-items: center;">
                        <button onclick="applyCropToCurrentPage()" class="btn-secondary">
                            <i class="fas fa-check"></i> Apply to This Page
                        </button>
                        <button onclick="applyCropToAllPages()" class="btn-primary">
                            <i class="fas fa-check-double"></i> Apply to All Pages
                        </button>
                        <button onclick="resetCropArea()" class="btn-secondary">
                            <i class="fas fa-undo"></i> Reset Selection
                        </button>
                        <div style="margin-left: auto; padding: 0.5rem 1rem; background: var(--bg-tertiary); border-radius: 6px; font-size: 0.85rem; color: var(--text-secondary);">
                            <i class="fas fa-keyboard"></i> 
                            <strong>Shortcuts:</strong> 
                            ← → (Navigate) | 
                            Enter (Apply) | 
                            Shift+Enter (Apply All) | 
                            Esc (Reset)
                        </div>
                    </div>
                </div>

                <div id="cropped-pages-preview" class="hidden" style="background: white; padding: 1.5rem; border-radius: 12px;">
                    <h3>📄 Pages to be cropped (<span id="cropped-count">0</span>)</h3>
                    <div id="cropped-pages-list" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 1rem; margin-top: 1rem;"></div>
                </div>
            </div>

            <div id="crop-actions" class="action-buttons hidden">
                <button onclick="processCroppedPdf()" class="btn-primary" style="padding: 1rem 2rem; font-size: 1.1rem;">
                    <i class="fas fa-crop-alt"></i> Generate Cropped PDF
                </button>
            </div>

            <div id="crop-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('crop-upload-zone').onclick = () => {
        document.getElementById('crop-pdf-input').click();
    };

    document.getElementById('crop-pdf-input').addEventListener('change', handleCropPdfUpload);
};

async function handleCropPdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        cropPdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        cropPdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        const pageCount = cropPdfDoc.getPageCount();
        currentCropPage = 1;
        pdfPages = Array(pageCount).fill(null).map(() => ({ cropped: false, crop: null }));
        
        document.getElementById('crop-editor').classList.remove('hidden');
        await renderCropPage();
        setupCropInteraction();
        setupKeyboardShortcuts();
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

function setupKeyboardShortcuts() {
    document.addEventListener('keydown', handleCropKeyPress);
}

function handleCropKeyPress(e) {
    // Only handle if crop editor is visible
    if (document.getElementById('crop-editor').classList.contains('hidden')) return;
    
    switch(e.key) {
        case 'ArrowLeft':
            e.preventDefault();
            prevCropPage();
            break;
        case 'ArrowRight':
            e.preventDefault();
            nextCropPage();
            break;
        case 'Enter':
            e.preventDefault();
            if (e.shiftKey) {
                applyCropToAllPages();
            } else {
                applyCropToCurrentPage();
            }
            break;
        case 'Escape':
            e.preventDefault();
            resetCropArea();
            break;
    }
}

async function renderCropPage() {
    const loadingTask = pdfjsLib.getDocument(await cropPdfFile.arrayBuffer());
    const pdf = await loadingTask.promise;
    const page = await pdf.getPage(currentCropPage);
    
    const canvas = document.getElementById('crop-canvas');
    const ctx = canvas.getContext('2d');
    
    // Scale to fit container (max 800px width)
    const maxWidth = Math.min(800, window.innerWidth - 100);
    const viewport = page.getViewport({ scale: 1 });
    canvasScale = maxWidth / viewport.width;
    const scaledViewport = page.getViewport({ scale: canvasScale });
    
    canvas.width = scaledViewport.width;
    canvas.height = scaledViewport.height;
    
    await page.render({ canvasContext: ctx, viewport: scaledViewport }).promise;
    
    // Update page info
    const pageCount = cropPdfDoc.getPageCount();
    document.getElementById('crop-page-info').textContent = `Page ${currentCropPage} / ${pageCount}`;
    document.getElementById('prev-page-btn').disabled = currentCropPage === 1;
    document.getElementById('next-page-btn').disabled = currentCropPage === pageCount;
    
    // Show existing crop if any
    if (pdfPages[currentCropPage - 1].cropped) {
        const existingCrop = pdfPages[currentCropPage - 1].crop;
        showCropOverlay(existingCrop.x, existingCrop.y, existingCrop.width, existingCrop.height);
        cropData = { ...existingCrop };
    } else {
        document.getElementById('crop-overlay').style.display = 'none';
        cropData = { x: 0, y: 0, width: 0, height: 0 };
    }
}

function setupCropInteraction() {
    const canvas = document.getElementById('crop-canvas');
    const overlay = document.getElementById('crop-overlay');
    
    function getCoordinates(e, rect) {
        const clientX = e.touches ? e.touches[0].clientX : e.clientX;
        const clientY = e.touches ? e.touches[0].clientY : e.clientY;
        return {
            x: clientX - rect.left,
            y: clientY - rect.top
        };
    }
    
    function startDrag(e) {
        e.preventDefault();
        const rect = canvas.getBoundingClientRect();
        const coords = getCoordinates(e, rect);
        dragStart.x = coords.x;
        dragStart.y = coords.y;
        isDragging = true;
        
        cropData = { x: dragStart.x, y: dragStart.y, width: 0, height: 0 };
        overlay.style.display = 'block';
        canvas.style.cursor = 'crosshair';
    }
    
    function moveDrag(e) {
        if (!isDragging) return;
        e.preventDefault();
        
        const rect = canvas.getBoundingClientRect();
        const coords = getCoordinates(e, rect);
        
        cropData.width = coords.x - dragStart.x;
        cropData.height = coords.y - dragStart.y;
        
        showCropOverlay(cropData.x, cropData.y, cropData.width, cropData.height);
    }
    
    function endDrag() {
        if (!isDragging) return;
        isDragging = false;
        canvas.style.cursor = 'crosshair';
        
        // Normalize negative dimensions
        if (cropData.width < 0) {
            cropData.x += cropData.width;
            cropData.width = Math.abs(cropData.width);
        }
        if (cropData.height < 0) {
            cropData.y += cropData.height;
            cropData.height = Math.abs(cropData.height);
        }
        
        // Clamp to canvas bounds
        cropData.x = Math.max(0, Math.min(cropData.x, canvas.width));
        cropData.y = Math.max(0, Math.min(cropData.y, canvas.height));
        cropData.width = Math.min(cropData.width, canvas.width - cropData.x);
        cropData.height = Math.min(cropData.height, canvas.height - cropData.y);
        
        showCropOverlay(cropData.x, cropData.y, cropData.width, cropData.height);
    }
    
    // Mouse events
    canvas.onmousedown = startDrag;
    canvas.onmousemove = moveDrag;
    canvas.onmouseup = endDrag;
    canvas.onmouseleave = endDrag;
    
    // Touch events for mobile
    canvas.ontouchstart = startDrag;
    canvas.ontouchmove = moveDrag;
    canvas.ontouchend = endDrag;
    canvas.ontouchcancel = endDrag;
}

function showCropOverlay(x, y, width, height) {
    const overlay = document.getElementById('crop-overlay');
    const absWidth = Math.abs(width);
    const absHeight = Math.abs(height);
    
    overlay.style.left = (width < 0 ? x + width : x) + 'px';
    overlay.style.top = (height < 0 ? y + height : y) + 'px';
    overlay.style.width = absWidth + 'px';
    overlay.style.height = absHeight + 'px';
    overlay.style.display = 'block';
    
    // Update label with dimensions
    const label = overlay.querySelector('div');
    if (label && absWidth > 0 && absHeight > 0) {
        label.textContent = `${Math.round(absWidth)} × ${Math.round(absHeight)} px`;
    } else if (label) {
        label.textContent = 'Drag to select area';
    }
}

function resetCropArea() {
    document.getElementById('crop-overlay').style.display = 'none';
    cropData = { x: 0, y: 0, width: 0, height: 0 };
}

function applyCropToCurrentPage() {
    if (cropData.width === 0 || cropData.height === 0) {
        alert('Please select an area to crop');
        return;
    }
    
    // Store crop data for current page (convert to PDF coordinates)
    const canvas = document.getElementById('crop-canvas');
    pdfPages[currentCropPage - 1] = {
        cropped: true,
        crop: {
            x: cropData.x / canvasScale,
            y: (canvas.height - cropData.y - cropData.height) / canvasScale, // Flip Y coordinate
            width: cropData.width / canvasScale,
            height: cropData.height / canvasScale
        }
    };
    
    updateCroppedPagesPreview();
    
    // Show success feedback
    showToast(`✓ Crop applied to page ${currentCropPage}`);
    
    // Auto-advance to next page if available
    if (currentCropPage < cropPdfDoc.getPageCount()) {
        nextCropPage();
    }
}

function applyCropToAllPages() {
    if (cropData.width === 0 || cropData.height === 0) {
        alert('Please select an area to crop');
        return;
    }
    
    const canvas = document.getElementById('crop-canvas');
    const cropInfo = {
        cropped: true,
        crop: {
            x: cropData.x / canvasScale,
            y: (canvas.height - cropData.y - cropData.height) / canvasScale,
            width: cropData.width / canvasScale,
            height: cropData.height / canvasScale
        }
    };
    
    // Apply to all pages
    for (let i = 0; i < pdfPages.length; i++) {
        pdfPages[i] = { ...cropInfo };
    }
    
    updateCroppedPagesPreview();
    showToast(`✓ Crop applied to all ${pdfPages.length} pages!`);
}

function showToast(message) {
    const toast = document.createElement('div');
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: var(--success-color);
        color: white;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        box-shadow: var(--shadow-xl);
        z-index: 10000;
        animation: slideIn 0.3s ease;
        font-weight: 500;
    `;
    toast.innerHTML = `<i class="fas fa-check-circle"></i> ${message}`;
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => toast.remove(), 300);
    }, 2000);
}

function updateCroppedPagesPreview() {
    const croppedCount = pdfPages.filter(p => p.cropped).length;
    
    if (croppedCount === 0) {
        document.getElementById('cropped-pages-preview').classList.add('hidden');
        document.getElementById('crop-actions').classList.add('hidden');
        return;
    }
    
    document.getElementById('cropped-pages-preview').classList.remove('hidden');
    document.getElementById('crop-actions').classList.remove('hidden');
    document.getElementById('cropped-count').textContent = croppedCount;
    
    const list = document.getElementById('cropped-pages-list');
    list.innerHTML = '';
    
    pdfPages.forEach((page, index) => {
        if (page.cropped) {
            const item = document.createElement('div');
            item.style.cssText = 'padding: 1rem; background: var(--bg-secondary); border-radius: 8px; text-align: center; border: 2px solid var(--primary-color);';
            item.innerHTML = `
                <i class="fas fa-file-pdf" style="font-size: 2rem; color: var(--primary-color); margin-bottom: 0.5rem;"></i>
                <div style="font-weight: 500;">Page ${index + 1}</div>
                <button onclick="removeCropFromPage(${index})" class="btn-icon" style="margin-top: 0.5rem;">
                    <i class="fas fa-times"></i> Remove
                </button>
            `;
            list.appendChild(item);
        }
    });
}

function removeCropFromPage(pageIndex) {
    pdfPages[pageIndex] = { cropped: false, crop: null };
    updateCroppedPagesPreview();
    
    if (pageIndex + 1 === currentCropPage) {
        renderCropPage();
    }
}

function prevCropPage() {
    if (currentCropPage > 1) {
        currentCropPage--;
        renderCropPage();
    }
}

function nextCropPage() {
    if (currentCropPage < cropPdfDoc.getPageCount()) {
        currentCropPage++;
        renderCropPage();
    }
}

async function processCroppedPdf() {
    const croppedCount = pdfPages.filter(p => p.cropped).length;
    
    if (croppedCount === 0) {
        alert('Please crop at least one page');
        return;
    }
    
    const resultDiv = document.getElementById('crop-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Cropping PDF...</h3><p>Processing ' + croppedCount + ' page(s)</p></div>';
    resultDiv.classList.remove('hidden');
    
    try {
        const newPdf = await PDFLib.PDFDocument.create();
        
        for (let i = 0; i < pdfPages.length; i++) {
            const [copiedPage] = await newPdf.copyPages(cropPdfDoc, [i]);
            
            if (pdfPages[i].cropped) {
                const crop = pdfPages[i].crop;
                
                // Set crop box to the selected area
                copiedPage.setCropBox(
                    crop.x,
                    crop.y,
                    crop.width,
                    crop.height
                );
            }
            
            newPdf.addPage(copiedPage);
        }
        
        const pdfBytes = await newPdf.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);
        const size = (blob.size / 1024).toFixed(1);
        
        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ PDF Cropped Successfully!</h3>
                <p>${croppedCount} of ${pdfPages.length} pages cropped • ${size} KB</p>
                <div class="button-group">
                    <a href="${url}" download="cropped.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download Cropped PDF
                    </a>
                    <button onclick="location.reload()" class="btn-secondary">
                        <i class="fas fa-redo"></i> Crop Another PDF
                    </button>
                </div>
            </div>
        `;
        
        // Scroll to result
        resultDiv.scrollIntoView({ behavior: 'smooth', block: 'center' });
    } catch (error) {
        console.error('Crop error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
