// Compress PDF - Fast client-side compression
let compressPdfFile = null;
let compressPdfDoc = null;

window.initializeCompressPdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-compress-alt"></i> Compress PDF</h2>
                <p>Reduce PDF file size while maintaining quality</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="compress-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <p>Compress your PDF to reduce file size</p>
                    <input type="file" id="compress-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="compress-options" class="split-options hidden">
                <h3>Compression Level</h3>
                <div class="split-mode-selector">
                    <button class="split-mode-btn active" onclick="setCompressionLevel('recommended')" id="level-recommended">
                        <i class="fas fa-balance-scale"></i>
                        <div>Recommended</div>
                        <small>Best balance</small>
                    </button>
                    <button class="split-mode-btn" onclick="setCompressionLevel('extreme')" id="level-extreme">
                        <i class="fas fa-compress-arrows-alt"></i>
                        <div>Extreme</div>
                        <small>Smallest size</small>
                    </button>
                    <button class="split-mode-btn" onclick="setCompressionLevel('low')" id="level-low">
                        <i class="fas fa-star"></i>
                        <div>Low</div>
                        <small>Best quality</small>
                    </button>
                </div>
                
                <div style="margin-top: 1.5rem; padding: 1rem; background: var(--bg-secondary); border-radius: 8px;">
                    <div style="display: flex; justify-content: space-between; margin-bottom: 0.5rem;">
                        <span style="font-weight: 500;">Original Size:</span>
                        <span id="original-size">--</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="font-weight: 500;">Estimated Size:</span>
                        <span id="estimated-size" style="color: var(--success-color); font-weight: 600;">--</span>
                    </div>
                </div>
            </div>

            <div id="compress-actions" class="action-buttons hidden">
                <button onclick="compressPdfNow()" class="btn-primary">
                    <i class="fas fa-compress-alt"></i> Compress PDF
                </button>
            </div>

            <div id="compress-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('compress-upload-zone').onclick = () => {
        document.getElementById('compress-pdf-input').click();
    };

    document.getElementById('compress-pdf-input').addEventListener('change', handleCompressPdfUpload);
};

let compressionLevel = 'recommended';

async function handleCompressPdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        compressPdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        compressPdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        document.getElementById('compress-options').classList.remove('hidden');
        document.getElementById('compress-actions').classList.remove('hidden');
        
        const sizeMB = (file.size / 1024 / 1024).toFixed(2);
        document.getElementById('original-size').textContent = sizeMB + ' MB';
        updateEstimatedSize();
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

function setCompressionLevel(level) {
    compressionLevel = level;
    document.querySelectorAll('.split-mode-btn').forEach(btn => btn.classList.remove('active'));
    document.getElementById('level-' + level).classList.add('active');
    updateEstimatedSize();
}

function updateEstimatedSize() {
    if (!compressPdfFile) return;
    
    const originalSize = compressPdfFile.size / 1024 / 1024;
    let reduction = 0.5; // 50% reduction
    
    if (compressionLevel === 'extreme') reduction = 0.7;
    if (compressionLevel === 'low') reduction = 0.3;
    
    const estimatedSize = (originalSize * (1 - reduction)).toFixed(2);
    const percentage = (reduction * 100).toFixed(0);
    
    document.getElementById('estimated-size').textContent = `${estimatedSize} MB (~${percentage}% smaller)`;
}

async function compressPdfNow() {
    if (!compressPdfDoc) return;

    const resultDiv = document.getElementById('compress-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Compressing PDF...</h3><p>Optimizing file size</p></div>';
    resultDiv.classList.remove('hidden');

    try {
        // Create optimized PDF
        const pdfBytes = await compressPdfDoc.save({
            useObjectStreams: true,
            addDefaultPage: false,
            objectsPerTick: compressionLevel === 'extreme' ? 100 : 50
        });
        
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);
        
        const originalSize = (compressPdfFile.size / 1024 / 1024).toFixed(2);
        const newSize = (blob.size / 1024 / 1024).toFixed(2);
        const reduction = (((compressPdfFile.size - blob.size) / compressPdfFile.size) * 100).toFixed(1);

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ PDF Compressed Successfully!</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1rem; margin: 1.5rem 0; text-align: center;">
                    <div style="padding: 1rem; background: var(--bg-secondary); border-radius: 8px;">
                        <div style="font-size: 0.9rem; color: var(--text-secondary); margin-bottom: 0.5rem;">Original</div>
                        <div style="font-size: 1.5rem; font-weight: 600;">${originalSize} MB</div>
                    </div>
                    <div style="padding: 1rem; background: var(--success-color); color: white; border-radius: 8px;">
                        <div style="font-size: 0.9rem; opacity: 0.9; margin-bottom: 0.5rem;">Compressed</div>
                        <div style="font-size: 1.5rem; font-weight: 600;">${newSize} MB</div>
                    </div>
                    <div style="padding: 1rem; background: var(--primary-color); color: white; border-radius: 8px;">
                        <div style="font-size: 0.9rem; opacity: 0.9; margin-bottom: 0.5rem;">Reduced</div>
                        <div style="font-size: 1.5rem; font-weight: 600;">${reduction}%</div>
                    </div>
                </div>
                <div class="button-group">
                    <a href="${url}" download="compressed.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download Compressed PDF
                    </a>
                    <button onclick="location.reload()" class="btn-secondary">
                        <i class="fas fa-redo"></i> Compress Another
                    </button>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Compression error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
