// Compress Image Tool

let imageToCompress = null;

window.initializeCompressImage = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-compress"></i> Compress Image</h2>
                <p>Reduce image file size</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('compress-image-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop image here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP</p>
                <input type="file" id="compress-image-input" accept="image/*" style="display: none;">
            </div>

            <div id="compress-image-preview" class="preview-section hidden">
                <img id="compress-preview" style="max-width: 100%;">
            </div>

            <div id="compress-image-options" class="options-section hidden">
                <h3>Compression Settings</h3>
                
                <div class="form-group">
                    <label>Quality: <span id="quality-value">80</span>%</label>
                    <input type="range" id="compress-quality" min="1" max="100" value="80" oninput="updateQualityValue()">
                </div>

                <div class="form-group">
                    <label>Max Width (pixels):</label>
                    <input type="number" id="max-width" class="form-input" placeholder="Leave empty for original">
                </div>

                <div class="form-group">
                    <label>Max Height (pixels):</label>
                    <input type="number" id="max-height" class="form-input" placeholder="Leave empty for original">
                </div>

                <button onclick="compressImage()" class="btn-primary">
                    <i class="fas fa-compress"></i> Compress Image
                </button>
            </div>

            <div id="compress-image-result" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('compress-image-input').addEventListener('change', handleCompressImageUpload);
};

function handleCompressImageUpload(e) {
    imageToCompress = e.target.files[0];
    if (!imageToCompress) return;

    const preview = document.getElementById('compress-preview');
    preview.src = URL.createObjectURL(imageToCompress);
    
    document.getElementById('compress-image-preview').classList.remove('hidden');
    document.getElementById('compress-image-options').classList.remove('hidden');
}

function updateQualityValue() {
    document.getElementById('quality-value').textContent = document.getElementById('compress-quality').value;
}

async function compressImage() {
    if (!imageToCompress) return;

    const resultDiv = document.getElementById('compress-image-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Compressing image...</div>';
    resultDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('image', imageToCompress);
    formData.append('quality', document.getElementById('compress-quality').value);
    
    const maxWidth = document.getElementById('max-width').value;
    const maxHeight = document.getElementById('max-height').value;
    if (maxWidth) formData.append('maxWidth', maxWidth);
    if (maxHeight) formData.append('maxHeight', maxHeight);

    try {
        const response = await fetch('/api/image/compress', {
            method: 'POST',
            body: formData
        });

        const blob = await response.blob();
        const url = URL.createObjectURL(blob);
        
        const originalSize = (imageToCompress.size / 1024).toFixed(2);
        const compressedSize = (blob.size / 1024).toFixed(2);
        const reduction = (((imageToCompress.size - blob.size) / imageToCompress.size) * 100).toFixed(1);

        resultDiv.innerHTML = `
            <h3>Image Compressed!</h3>
            <img src="${url}" style="max-width: 300px;">
            <p>Original: ${originalSize} KB</p>
            <p>Compressed: ${compressedSize} KB</p>
            <p>Reduction: ${reduction}%</p>
            <a href="${url}" download="compressed.jpg" class="btn-primary">
                <i class="fas fa-download"></i> Download
            </a>
        `;
    } catch (error) {
        resultDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
