// Color Palette Extractor Tool
function initializeColorExtractor() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="color-extractor-container">
            <h2>🎨 Color Palette Extractor</h2>
            <p class="tool-description">Extract dominant color palettes from images</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="color-file-input" accept="image/*" onchange="extractColors(this)">
                    <span>📁 Choose Image to Analyze</span>
                </label>
            </div>

            <div id="color-preview-section" class="preview-section hidden">
                <h3>Image Preview</h3>
                <img id="color-preview" src="" alt="Preview" style="max-width: 100%; border-radius: 8px;">
            </div>

            <div id="color-results" class="color-results hidden">
                <h3>Extracted Color Palette</h3>
                <div id="color-palette" class="color-palette-grid"></div>
                
                <div class="export-section">
                    <h4>Export Palette</h4>
                    <div class="button-group">
                        <button class="btn btn-secondary" onclick="exportColorPalette('css')">📄 Export as CSS</button>
                        <button class="btn btn-secondary" onclick="exportColorPalette('scss')">📄 Export as SCSS</button>
                        <button class="btn btn-secondary" onclick="exportColorPalette('json')">📄 Export as JSON</button>
                        <button class="btn btn-secondary" onclick="copyAllColors()">📋 Copy All Hex</button>
                    </div>
                </div>
            </div>
        </div>
    `;
}

let extractedColors = [];

async function extractColors(input) {
    const file = input.files[0];
    if (!file) return;

    // Preview
    const reader = new FileReader();
    reader.onload = function(e) {
        document.getElementById('color-preview').src = e.target.result;
        document.getElementById('color-preview-section').classList.remove('hidden');
    };
    reader.readAsDataURL(file);

    // Extract colors via API
    const formData = new FormData();
    formData.append('image', file);

    try {
        const response = await fetch('/api/utility/color-extract', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            extractedColors = data.colors;
            displayColorPalette(data.colors);
        } else {
            alert('Failed to extract colors: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while extracting colors');
    }
}

function displayColorPalette(colors) {
    const palette = document.getElementById('color-palette');
    palette.innerHTML = '';

    colors.forEach((color, index) => {
        const colorCard = document.createElement('div');
        colorCard.className = 'color-card';
        
        // Determine if text should be white or black based on brightness
        const brightness = (color.r * 299 + color.g * 587 + color.b * 114) / 1000;
        const textColor = brightness > 128 ? '#000' : '#fff';
        
        colorCard.innerHTML = `
            <div class="color-swatch" style="background-color: ${color.hex};">
                <span style="color: ${textColor}; font-weight: bold;">${color.hex}</span>
            </div>
            <div class="color-info">
                <div class="color-values">
                    <span><strong>HEX:</strong> ${color.hex}</span>
                    <span><strong>RGB:</strong> ${color.r}, ${color.g}, ${color.b}</span>
                </div>
                <div class="color-actions">
                    <button onclick="copyColor('${color.hex}')" class="btn-small">📋 Copy HEX</button>
                    <button onclick="copyColor('rgb(${color.r}, ${color.g}, ${color.b})')" class="btn-small">📋 Copy RGB</button>
                </div>
            </div>
        `;
        palette.appendChild(colorCard);
    });

    document.getElementById('color-results').classList.remove('hidden');
}

function copyColor(value) {
    navigator.clipboard.writeText(value).then(() => {
        const btn = event.target;
        const originalText = btn.textContent;
        btn.textContent = '✅ Copied!';
        setTimeout(() => btn.textContent = originalText, 1500);
    });
}

function copyAllColors() {
    const hexColors = extractedColors.map(c => c.hex).join(', ');
    navigator.clipboard.writeText(hexColors).then(() => {
        const btn = event.target;
        const originalText = btn.textContent;
        btn.textContent = '✅ Copied All!';
        setTimeout(() => btn.textContent = originalText, 2000);
    });
}

function exportColorPalette(format) {
    let content = '';
    const timestamp = new Date().toISOString().split('T')[0];
    let filename = '';

    switch(format) {
        case 'css':
            content = ':root {\n';
            extractedColors.forEach((color, i) => {
                content += `  --color-${i + 1}: ${color.hex};\n`;
            });
            content += '}\n';
            filename = `color-palette-${timestamp}.css`;
            break;

        case 'scss':
            extractedColors.forEach((color, i) => {
                content += `$color-${i + 1}: ${color.hex};\n`;
            });
            filename = `color-palette-${timestamp}.scss`;
            break;

        case 'json':
            const jsonData = {
                palette: extractedColors.map((color, i) => ({
                    name: `color-${i + 1}`,
                    hex: color.hex,
                    rgb: { r: color.r, g: color.g, b: color.b }
                })),
                extracted: new Date().toISOString()
            };
            content = JSON.stringify(jsonData, null, 2);
            filename = `color-palette-${timestamp}.json`;
            break;
    }

    const blob = new Blob([content], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    link.click();
    
    URL.revokeObjectURL(url);
}


// Expose to window for router
window.initializeColorExtractor = initializeColorExtractor;
