// Collage Maker Tool
function initializeCollageMaker() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="collage-maker-container">
            <h2>🖼️ Collage Maker</h2>
            <p class="tool-description">Create beautiful photo collages from multiple images</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="collage-file-input" accept="image/*" multiple onchange="loadCollageImages(this)">
                    <span>📁 Choose Images (2-9 images)</span>
                </label>
            </div>

            <div id="collage-options" class="collage-options hidden">
                <div class="image-count-info">
                    <strong><span id="image-count">0</span> images selected</strong>
                </div>

                <h3>Collage Settings</h3>
                
                <div class="form-group">
                    <label>Layout Style:</label>
                    <select id="collage-layout" onchange="updateCollagePreview()">
                        <option value="grid">Grid Layout</option>
                        <option value="horizontal">Horizontal Strip</option>
                        <option value="vertical">Vertical Strip</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Spacing: <span id="spacing-value">10</span>px</label>
                    <input type="range" id="collage-spacing" min="0" max="50" value="10" onchange="updateSpacingValue()">
                </div>

                <div class="form-group">
                    <label>Background Color:</label>
                    <input type="color" id="collage-bg-color" value="#ffffff">
                </div>

                <div class="form-group">
                    <label>Output Width: <span id="width-value">1200</span>px</label>
                    <input type="range" id="collage-width" min="600" max="3000" value="1200" step="100" onchange="updateWidthValue()">
                </div>

                <div class="form-group">
                    <label>Image Quality:</label>
                    <select id="collage-quality">
                        <option value="80">Standard (80%)</option>
                        <option value="90" selected>High (90%)</option>
                        <option value="95">Very High (95%)</option>
                        <option value="100">Maximum (100%)</option>
                    </select>
                </div>

                <div id="image-preview-grid" class="image-grid"></div>

                <button class="btn btn-primary" onclick="createCollage()">Create Collage</button>
            </div>

            <div id="collage-result" class="result-section hidden">
                <h3>Your Collage</h3>
                <img id="collage-preview" src="" alt="Collage" style="max-width: 100%; border-radius: 8px;">
                <div class="button-group">
                    <button class="btn btn-secondary" onclick="downloadCollage()">💾 Download Collage</button>
                    <button class="btn btn-secondary" onclick="resetCollage()">🔄 Create New</button>
                </div>
            </div>
        </div>
    `;
}

let collageImages = [];
let collageResultUrl = '';

function loadCollageImages(input) {
    collageImages = Array.from(input.files);
    
    if (collageImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }
    
    if (collageImages.length > 9) {
        alert('Maximum 9 images allowed. Only the first 9 will be used.');
        collageImages = collageImages.slice(0, 9);
    }

    document.getElementById('image-count').textContent = collageImages.length;
    document.getElementById('collage-options').classList.remove('hidden');
    
    displayImagePreviewGrid();
}

function displayImagePreviewGrid() {
    const grid = document.getElementById('image-preview-grid');
    grid.innerHTML = '<h4>Selected Images</h4>';
    
    collageImages.forEach((file, index) => {
        const reader = new FileReader();
        reader.onload = function(e) {
            const div = document.createElement('div');
            div.className = 'preview-item';
            div.innerHTML = `
                <img src="${e.target.result}" alt="Image ${index + 1}">
                <button onclick="removeCollageImage(${index})" class="btn-remove">✕</button>
            `;
            grid.appendChild(div);
        };
        reader.readAsDataURL(file);
    });
}

function removeCollageImage(index) {
    collageImages.splice(index, 1);
    document.getElementById('image-count').textContent = collageImages.length;
    
    if (collageImages.length < 2) {
        document.getElementById('collage-options').classList.add('hidden');
    } else {
        displayImagePreviewGrid();
    }
}

function updateSpacingValue() {
    document.getElementById('spacing-value').textContent = 
        document.getElementById('collage-spacing').value;
}

function updateWidthValue() {
    document.getElementById('width-value').textContent = 
        document.getElementById('collage-width').value;
}

function updateCollagePreview() {
    // Optional: Show layout preview
}

async function createCollage() {
    if (collageImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }

    const formData = new FormData();
    
    collageImages.forEach(file => {
        formData.append('images', file);
    });
    
    formData.append('layout', document.getElementById('collage-layout').value);
    formData.append('spacing', document.getElementById('collage-spacing').value);
    formData.append('bgColor', document.getElementById('collage-bg-color').value);
    formData.append('width', document.getElementById('collage-width').value);
    formData.append('quality', document.getElementById('collage-quality').value);

    try {
        const response = await fetch('/api/creative/collage-create', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            collageResultUrl = data.url;
            document.getElementById('collage-preview').src = data.url;
            document.getElementById('collage-result').classList.remove('hidden');
            
            // Scroll to result
            document.getElementById('collage-result').scrollIntoView({ behavior: 'smooth' });
        } else {
            alert('Failed to create collage: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while creating the collage');
    }
}

function downloadCollage() {
    if (!collageResultUrl) return;
    
    const link = document.createElement('a');
    link.href = collageResultUrl;
    link.download = 'collage_' + Date.now() + '.jpg';
    link.click();
}

function resetCollage() {
    collageImages = [];
    document.getElementById('collage-file-input').value = '';
    document.getElementById('collage-options').classList.add('hidden');
    document.getElementById('collage-result').classList.add('hidden');
}


// Expose to window for router
window.initializeCollageMaker = initializeCollageMaker;
