// Changelog Generator

let commits = [];

window.initializeChangelogGenerator = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-list-alt"></i> Changelog Generator</h2>
                <p>Generate formatted changelogs from commit messages</p>
            </div>

            <div class="changelog-form">
                <h3>Project Information</h3>
                
                <div class="form-group">
                    <label>Project Name:</label>
                    <input type="text" id="changelog-project-name" class="form-input" placeholder="My Awesome Project">
                </div>

                <div class="form-group">
                    <label>Version:</label>
                    <input type="text" id="changelog-version" class="form-input" placeholder="1.0.0">
                </div>

                <div class="form-group">
                    <label>Release Date:</label>
                    <input type="date" id="changelog-date" class="form-input">
                </div>

                <h3>Changes</h3>
                
                <div id="changelog-entries">
                    <div class="changelog-entry">
                        <select class="change-type">
                            <option value="Added">Added</option>
                            <option value="Changed">Changed</option>
                            <option value="Deprecated">Deprecated</option>
                            <option value="Removed">Removed</option>
                            <option value="Fixed">Fixed</option>
                            <option value="Security">Security</option>
                        </select>
                        <input type="text" class="change-description form-input" placeholder="Description of change">
                        <button onclick="removeChangeEntry(this)" class="btn-small btn-danger">×</button>
                    </div>
                </div>

                <button onclick="addChangeEntry()" class="btn-secondary">
                    <i class="fas fa-plus"></i> Add Entry
                </button>

                <h3>Format</h3>
                
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="changelog-format">
                        <option value="markdown">Markdown</option>
                        <option value="keepachangelog">Keep a Changelog</option>
                        <option value="html">HTML</option>
                        <option value="text">Plain Text</option>
                    </select>
                </div>

                <button onclick="generateChangelog()" class="btn-primary">
                    <i class="fas fa-file-alt"></i> Generate Changelog
                </button>

                <div id="changelog-output" class="results-section hidden">
                    <label>Generated Changelog:</label>
                    <textarea id="changelog-text" rows="20" readonly></textarea>
                    <div class="button-group">
                        <button onclick="copyChangelog()" class="btn-secondary">
                            <i class="fas fa-copy"></i> Copy
                        </button>
                        <button onclick="downloadChangelog()" class="btn-primary">
                            <i class="fas fa-download"></i> Download
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Set today's date
    document.getElementById('changelog-date').value = new Date().toISOString().split('T')[0];
};

function addChangeEntry() {
    const container = document.getElementById('changelog-entries');
    const entry = document.createElement('div');
    entry.className = 'changelog-entry';
    entry.innerHTML = `
        <select class="change-type">
            <option value="Added">Added</option>
            <option value="Changed">Changed</option>
            <option value="Deprecated">Deprecated</option>
            <option value="Removed">Removed</option>
            <option value="Fixed">Fixed</option>
            <option value="Security">Security</option>
        </select>
        <input type="text" class="change-description form-input" placeholder="Description of change">
        <button onclick="removeChangeEntry(this)" class="btn-small btn-danger">×</button>
    `;
    container.appendChild(entry);
}

function removeChangeEntry(btn) {
    if (document.querySelectorAll('.changelog-entry').length > 1) {
        btn.closest('.changelog-entry').remove();
    } else {
        alert('At least one entry is required');
    }
}

function generateChangelog() {
    const projectName = document.getElementById('changelog-project-name').value || 'Project';
    const version = document.getElementById('changelog-version').value || '1.0.0';
    const date = document.getElementById('changelog-date').value || new Date().toISOString().split('T')[0];
    const format = document.getElementById('changelog-format').value;

    const entries = Array.from(document.querySelectorAll('.changelog-entry')).map(entry => ({
        type: entry.querySelector('.change-type').value,
        description: entry.querySelector('.change-description').value.trim()
    })).filter(e => e.description);

    if (entries.length === 0) {
        alert('Please add at least one change entry');
        return;
    }

    // Group by type
    const grouped = {};
    entries.forEach(entry => {
        if (!grouped[entry.type]) {
            grouped[entry.type] = [];
        }
        grouped[entry.type].push(entry.description);
    });

    let changelog = '';

    if (format === 'markdown' || format === 'keepachangelog') {
        changelog = `# ${projectName} Changelog\n\n`;
        changelog += `## [${version}] - ${date}\n\n`;
        
        Object.keys(grouped).forEach(type => {
            changelog += `### ${type}\n\n`;
            grouped[type].forEach(desc => {
                changelog += `- ${desc}\n`;
            });
            changelog += '\n';
        });
    } else if (format === 'html') {
        changelog = `<h1>${projectName} Changelog</h1>\n`;
        changelog += `<h2>Version ${version} - ${date}</h2>\n`;
        
        Object.keys(grouped).forEach(type => {
            changelog += `<h3>${type}</h3>\n<ul>\n`;
            grouped[type].forEach(desc => {
                changelog += `  <li>${desc}</li>\n`;
            });
            changelog += `</ul>\n`;
        });
    } else {
        changelog = `${projectName} Changelog\n`;
        changelog += `Version ${version} - ${date}\n\n`;
        
        Object.keys(grouped).forEach(type => {
            changelog += `${type}:\n`;
            grouped[type].forEach(desc => {
                changelog += `  - ${desc}\n`;
            });
            changelog += '\n';
        });
    }

    document.getElementById('changelog-text').value = changelog;
    document.getElementById('changelog-output').classList.remove('hidden');
}

function copyChangelog() {
    const text = document.getElementById('changelog-text');
    text.select();
    document.execCommand('copy');
    alert('Changelog copied to clipboard!');
}

function downloadChangelog() {
    const text = document.getElementById('changelog-text').value;
    const format = document.getElementById('changelog-format').value;
    
    let filename = 'CHANGELOG';
    let mimeType = 'text/plain';
    
    if (format === 'markdown' || format === 'keepachangelog') {
        filename += '.md';
    } else if (format === 'html') {
        filename += '.html';
        mimeType = 'text/html';
    } else {
        filename += '.txt';
    }

    const blob = new Blob([text], { type: mimeType });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = filename;
    link.click();
}
