// Bulk File Renamer Tool
function initializeBulkRename() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="bulk-rename-container">
            <h2>📝 Bulk File Renamer</h2>
            <p class="tool-description">Rename multiple files at once with custom patterns</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="rename-file-input" multiple onchange="loadFilesForRename(this)">
                    <span>📁 Choose Files to Rename</span>
                </label>
            </div>

            <div id="rename-options" class="rename-options hidden">
                <h3>Rename Pattern</h3>
                
                <div class="form-group">
                    <label>Naming Pattern:</label>
                    <select id="rename-pattern" onchange="updateRenamePreview()">
                        <option value="prefix">Prefix + Original Name</option>
                        <option value="suffix">Original Name + Suffix</option>
                        <option value="replace">Replace Text</option>
                        <option value="sequential">Sequential Numbering</option>
                        <option value="custom">Custom Pattern</option>
                    </select>
                </div>

                <div id="pattern-inputs">
                    <div class="form-group">
                        <label>Prefix Text:</label>
                        <input type="text" id="rename-prefix" placeholder="e.g., IMG_" onkeyup="updateRenamePreview()">
                    </div>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="preserve-extension" checked onchange="updateRenamePreview()">
                        Preserve file extensions
                    </label>
                </div>

                <h3>File List & Preview</h3>
                <div id="rename-preview-list" class="rename-list"></div>

                <button class="btn btn-primary" onclick="performBulkRename()">Rename & Download All</button>
            </div>
        </div>
    `;
}

let renameFiles = [];

function loadFilesForRename(input) {
    renameFiles = Array.from(input.files);
    if (renameFiles.length === 0) return;

    document.getElementById('rename-options').classList.remove('hidden');
    updateRenamePreview();
}

function updateRenamePreview() {
    if (renameFiles.length === 0) return;

    const pattern = document.getElementById('rename-pattern').value;
    const preserveExt = document.getElementById('preserve-extension').checked;
    
    // Update pattern inputs based on selection
    updatePatternInputs(pattern);

    const previewList = document.getElementById('rename-preview-list');
    previewList.innerHTML = '';

    renameFiles.forEach((file, index) => {
        const newName = generateNewFilename(file.name, index, pattern, preserveExt);
        
        const row = document.createElement('div');
        row.className = 'rename-row';
        row.innerHTML = `
            <div class="original-name">${file.name}</div>
            <div class="arrow">→</div>
            <div class="new-name">${newName}</div>
        `;
        previewList.appendChild(row);
    });
}

function updatePatternInputs(pattern) {
    const inputsDiv = document.getElementById('pattern-inputs');
    
    switch(pattern) {
        case 'prefix':
            inputsDiv.innerHTML = `
                <div class="form-group">
                    <label>Prefix Text:</label>
                    <input type="text" id="rename-prefix" placeholder="e.g., IMG_" onkeyup="updateRenamePreview()">
                </div>`;
            break;
        case 'suffix':
            inputsDiv.innerHTML = `
                <div class="form-group">
                    <label>Suffix Text:</label>
                    <input type="text" id="rename-suffix" placeholder="e.g., _backup" onkeyup="updateRenamePreview()">
                </div>`;
            break;
        case 'replace':
            inputsDiv.innerHTML = `
                <div class="form-group">
                    <label>Find Text:</label>
                    <input type="text" id="rename-find" placeholder="Text to find" onkeyup="updateRenamePreview()">
                </div>
                <div class="form-group">
                    <label>Replace With:</label>
                    <input type="text" id="rename-replace" placeholder="Replacement text" onkeyup="updateRenamePreview()">
                </div>`;
            break;
        case 'sequential':
            inputsDiv.innerHTML = `
                <div class="form-group">
                    <label>Base Name:</label>
                    <input type="text" id="rename-basename" placeholder="e.g., File" value="File" onkeyup="updateRenamePreview()">
                </div>
                <div class="form-group">
                    <label>Start Number:</label>
                    <input type="number" id="rename-startnumber" value="1" min="0" onkeyup="updateRenamePreview()">
                </div>
                <div class="form-group">
                    <label>Number Padding:</label>
                    <input type="number" id="rename-padding" value="3" min="1" max="6" onkeyup="updateRenamePreview()">
                </div>`;
            break;
        case 'custom':
            inputsDiv.innerHTML = `
                <div class="form-group">
                    <label>Custom Pattern:</label>
                    <input type="text" id="rename-custom" placeholder="e.g., {name}_{date}_{index}" onkeyup="updateRenamePreview()">
                </div>
                <div class="info-text">
                    Available tags: {name}, {index}, {date}, {time}, {random}
                </div>`;
            break;
    }
}

function generateNewFilename(originalName, index, pattern, preserveExt) {
    let name = originalName;
    let ext = '';
    
    if (preserveExt) {
        const lastDot = originalName.lastIndexOf('.');
        if (lastDot > 0) {
            name = originalName.substring(0, lastDot);
            ext = originalName.substring(lastDot);
        }
    }

    let newName = '';

    switch(pattern) {
        case 'prefix':
            const prefix = document.getElementById('rename-prefix')?.value || '';
            newName = prefix + name;
            break;
        case 'suffix':
            const suffix = document.getElementById('rename-suffix')?.value || '';
            newName = name + suffix;
            break;
        case 'replace':
            const find = document.getElementById('rename-find')?.value || '';
            const replace = document.getElementById('rename-replace')?.value || '';
            newName = find ? name.replace(new RegExp(find, 'g'), replace) : name;
            break;
        case 'sequential':
            const baseName = document.getElementById('rename-basename')?.value || 'File';
            const startNum = parseInt(document.getElementById('rename-startnumber')?.value || '1');
            const padding = parseInt(document.getElementById('rename-padding')?.value || '3');
            const num = (startNum + index).toString().padStart(padding, '0');
            newName = `${baseName}_${num}`;
            break;
        case 'custom':
            const customPattern = document.getElementById('rename-custom')?.value || '{name}_{index}';
            const date = new Date().toISOString().split('T')[0];
            const time = new Date().toTimeString().split(' ')[0].replace(/:/g, '-');
            const random = Math.random().toString(36).substring(2, 8);
            
            newName = customPattern
                .replace('{name}', name)
                .replace('{index}', (index + 1).toString().padStart(3, '0'))
                .replace('{date}', date)
                .replace('{time}', time)
                .replace('{random}', random);
            break;
        default:
            newName = name;
    }

    return newName + ext;
}

async function performBulkRename() {
    if (renameFiles.length === 0) {
        alert('Please select files to rename');
        return;
    }

    const pattern = document.getElementById('rename-pattern').value;
    const preserveExt = document.getElementById('preserve-extension').checked;

    const formData = new FormData();
    
    renameFiles.forEach((file, index) => {
        const newName = generateNewFilename(file.name, index, pattern, preserveExt);
        formData.append('files', file);
        formData.append('newNames', newName);
    });

    try {
        const response = await fetch('/api/utility/bulk-rename', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            // Download all renamed files as ZIP
            downloadRenamedFiles(data.files);
        } else {
            alert('Failed to rename files: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while renaming files');
    }
}

function downloadRenamedFiles(files) {
    files.forEach(file => {
        // Create a temporary download link for each file
        setTimeout(() => {
            const link = document.createElement('a');
            link.href = file.url;
            link.download = file.filename;
            link.click();
        }, 100 * files.indexOf(file));
    });
}


// Expose to window for router
window.initializeBulkRename = initializeBulkRename;
