// Image Format Batch Converter Tool
function initializeBatchConverter() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="batch-converter-container">
            <h2>🔄 Batch Image Converter</h2>
            <p class="tool-description">Convert multiple images to any format in one click</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="batch-convert-input" accept="image/*" multiple onchange="loadBatchImages(this)">
                    <span>📁 Choose Images (Multiple)</span>
                </label>
            </div>

            <div id="batch-options" class="batch-options hidden">
                <div class="image-count-info">
                    <strong><span id="batch-count">0</span> images selected</strong>
                </div>

                <div class="format-selector">
                    <h3>Convert To:</h3>
                    <div class="format-buttons">
                        <button class="format-btn" data-format="jpg" onclick="selectFormat('jpg')">
                            <span class="format-icon">📸</span> JPG
                        </button>
                        <button class="format-btn" data-format="png" onclick="selectFormat('png')">
                            <span class="format-icon">🖼️</span> PNG
                        </button>
                        <button class="format-btn" data-format="webp" onclick="selectFormat('webp')">
                            <span class="format-icon">🌐</span> WebP
                        </button>
                        <button class="format-btn" data-format="gif" onclick="selectFormat('gif')">
                            <span class="format-icon">🎞️</span> GIF
                        </button>
                        <button class="format-btn" data-format="bmp" onclick="selectFormat('bmp')">
                            <span class="format-icon">🎨</span> BMP
                        </button>
                        <button class="format-btn" data-format="tiff" onclick="selectFormat('tiff')">
                            <span class="format-icon">📄</span> TIFF
                        </button>
                        <button class="format-btn" data-format="ico" onclick="selectFormat('ico')">
                            <span class="format-icon">⭐</span> ICO
                        </button>
                        <button class="format-btn" data-format="svg" onclick="selectFormat('svg')">
                            <span class="format-icon">📐</span> SVG
                        </button>
                    </div>
                </div>

                <div id="conversion-settings" class="conversion-settings hidden">
                    <h3>Conversion Settings</h3>
                    
                    <div class="form-group">
                        <label>Quality: <span id="batch-quality-val">90</span>%</label>
                        <input type="range" id="batch-quality" min="1" max="100" value="90" oninput="updateBatchQuality()">
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="batch-resize" onchange="toggleBatchResize()">
                            Resize during conversion
                        </label>
                    </div>

                    <div id="batch-resize-options" class="hidden">
                        <div class="form-row">
                            <div class="form-group">
                                <label>Max Width:</label>
                                <input type="number" id="batch-max-width" value="1920" min="1">
                            </div>
                            <div class="form-group">
                                <label>Max Height:</label>
                                <input type="number" id="batch-max-height" value="1080" min="1">
                            </div>
                        </div>
                    </div>
                </div>

                <button class="btn btn-primary" id="convert-batch-btn" onclick="convertBatch()" disabled>
                    Convert Images
                </button>
            </div>

            <div id="batch-results" class="result-section hidden">
                <h3>Converted Images</h3>
                <div class="conversion-stats">
                    <p><strong>Converted:</strong> <span id="converted-count">0</span>/<span id="total-count">0</span></p>
                    <p><strong>Total Size:</strong> <span id="total-size">0 KB</span></p>
                </div>
                <div id="batch-grid" class="results-grid"></div>
                <button class="btn btn-secondary" onclick="downloadAllConverted()">📦 Download All as ZIP</button>
            </div>
        </div>
    `;
}

let batchFiles = [];
let selectedFormat = null;

function loadBatchImages(input) {
    batchFiles = Array.from(input.files);
    document.getElementById('batch-count').textContent = batchFiles.length;
    document.getElementById('batch-options').classList.remove('hidden');
}

function selectFormat(format) {
    selectedFormat = format;
    document.querySelectorAll('.format-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.format === format);
    });
    document.getElementById('conversion-settings').classList.remove('hidden');
    document.getElementById('convert-batch-btn').disabled = false;
}

function updateBatchQuality() {
    document.getElementById('batch-quality-val').textContent = 
        document.getElementById('batch-quality').value;
}

function toggleBatchResize() {
    const checked = document.getElementById('batch-resize').checked;
    document.getElementById('batch-resize-options').classList.toggle('hidden', !checked);
}

async function convertBatch() {
    if (batchFiles.length === 0 || !selectedFormat) return;

    const formData = new FormData();
    batchFiles.forEach(file => formData.append('images', file));
    formData.append('format', selectedFormat);
    formData.append('quality', document.getElementById('batch-quality').value);

    if (document.getElementById('batch-resize').checked) {
        formData.append('maxWidth', document.getElementById('batch-max-width').value);
        formData.append('maxHeight', document.getElementById('batch-max-height').value);
    }

    try {
        const response = await fetch('/api/image/batch-convert', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayBatchResults(data.images);
        } else {
            alert('Failed to convert images: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred during batch conversion');
    }
}

function displayBatchResults(images) {
    const grid = document.getElementById('batch-grid');
    grid.innerHTML = '';

    let totalSize = 0;

    images.forEach(img => {
        totalSize += img.size || 0;
        
        const item = document.createElement('div');
        item.className = 'result-item';
        item.innerHTML = `
            <img src="${img.url}" alt="${img.filename}">
            <div class="result-info">
                <p>${img.filename}</p>
                <p>${formatFileSize(img.size)}</p>
                <button class="btn-small" onclick="downloadFile('${img.url}', '${img.filename}')">Download</button>
            </div>
        `;
        grid.appendChild(item);
    });

    document.getElementById('converted-count').textContent = images.length;
    document.getElementById('total-count').textContent = batchFiles.length;
    document.getElementById('total-size').textContent = formatFileSize(totalSize);
    document.getElementById('batch-results').classList.remove('hidden');
}

function downloadAllConverted() {
    alert('Downloading all converted images...');
}


// Expose to window for router
window.initializeBatchConverter = initializeBatchConverter;
