// Base64 Image Encoder/Decoder Tool
function initializeBase64Converter() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="base64-converter-container">
            <h2>🔐 Base64 Image Converter</h2>
            <p class="tool-description">Convert images to Base64 strings and vice versa</p>

            <div class="tabs">
                <button class="tab-btn active" onclick="switchBase64Tab('encode')">Encode to Base64</button>
                <button class="tab-btn" onclick="switchBase64Tab('decode')">Decode from Base64</button>
            </div>

            <div id="encode-tab" class="tab-content active">
                <div class="upload-section">
                    <label class="file-input-label">
                        <input type="file" id="base64-file-input" accept="image/*" onchange="encodeToBase64(this)">
                        <span>📁 Choose Image to Encode</span>
                    </label>
                </div>

                <div id="encode-result" class="result-section hidden">
                    <h3>Base64 String</h3>
                    <div class="preview-container">
                        <img id="encode-preview" src="" alt="Preview" style="max-width: 300px; max-height: 300px;">
                    </div>
                    <textarea id="base64-output" rows="10" readonly></textarea>
                    <div class="button-group">
                        <button class="btn btn-secondary" onclick="copyBase64()">📋 Copy Base64</button>
                        <button class="btn btn-secondary" onclick="downloadBase64Txt()">💾 Download as TXT</button>
                    </div>
                    <div class="info-text">
                        File size: <span id="encode-filesize"></span> | 
                        Base64 size: <span id="base64-size"></span>
                    </div>
                </div>
            </div>

            <div id="decode-tab" class="tab-content">
                <div class="form-group">
                    <label>Paste Base64 String:</label>
                    <textarea id="base64-input" rows="8" placeholder="Paste your Base64 string here (e.g., data:image/png;base64,iVBORw0K...)"></textarea>
                </div>
                <button class="btn btn-primary" onclick="decodeFromBase64()">Decode & Preview</button>

                <div id="decode-result" class="result-section hidden">
                    <h3>Decoded Image</h3>
                    <div class="preview-container">
                        <img id="decode-preview" src="" alt="Decoded Image">
                    </div>
                    <button class="btn btn-secondary" onclick="downloadDecodedImage()">💾 Download Image</button>
                </div>
            </div>
        </div>
    `;
}

function switchBase64Tab(tab) {
    const tabs = document.querySelectorAll('.tab-btn');
    const contents = document.querySelectorAll('.tab-content');
    
    tabs.forEach(t => t.classList.remove('active'));
    contents.forEach(c => c.classList.remove('active'));
    
    if (tab === 'encode') {
        tabs[0].classList.add('active');
        document.getElementById('encode-tab').classList.add('active');
    } else {
        tabs[1].classList.add('active');
        document.getElementById('decode-tab').classList.add('active');
    }
}

function encodeToBase64(input) {
    const file = input.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        const base64String = e.target.result;
        
        document.getElementById('encode-preview').src = base64String;
        document.getElementById('base64-output').value = base64String;
        document.getElementById('encode-filesize').textContent = formatFileSize(file.size);
        document.getElementById('base64-size').textContent = formatFileSize(base64String.length);
        
        document.getElementById('encode-result').classList.remove('hidden');
    };
    reader.readAsDataURL(file);
}

function decodeFromBase64() {
    const base64Input = document.getElementById('base64-input').value.trim();
    
    if (!base64Input) {
        alert('Please paste a Base64 string');
        return;
    }

    try {
        // Validate Base64 format
        const base64Data = base64Input.includes(',') ? base64Input : 'data:image/png;base64,' + base64Input;
        
        document.getElementById('decode-preview').src = base64Data;
        document.getElementById('decode-result').classList.remove('hidden');
        
        // Store for download
        window.decodedImageData = base64Data;
    } catch (error) {
        alert('Invalid Base64 string. Please check your input.');
    }
}

function copyBase64() {
    const textarea = document.getElementById('base64-output');
    textarea.select();
    document.execCommand('copy');
    
    const btn = event.target;
    const originalText = btn.textContent;
    btn.textContent = '✅ Copied!';
    setTimeout(() => btn.textContent = originalText, 2000);
}

function downloadBase64Txt() {
    const base64String = document.getElementById('base64-output').value;
    const blob = new Blob([base64String], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    
    const link = document.createElement('a');
    link.href = url;
    link.download = 'base64_encoded_' + Date.now() + '.txt';
    link.click();
    
    URL.revokeObjectURL(url);
}

function downloadDecodedImage() {
    if (!window.decodedImageData) return;
    
    const link = document.createElement('a');
    link.href = window.decodedImageData;
    
    // Determine file extension from MIME type
    const mimeMatch = window.decodedImageData.match(/data:image\/([^;]+);/);
    const ext = mimeMatch ? mimeMatch[1] : 'png';
    
    link.download = 'decoded_image_' + Date.now() + '.' + ext;
    link.click();
}

function formatFileSize(bytes) {
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(2) + ' KB';
    return (bytes / (1024 * 1024)).toFixed(2) + ' MB';
}

// Expose to window for router
window.initializeBase64Converter = initializeBase64Converter;
