// Advanced Background Remover

let bgRemoveImages = [];

window.initializeBackgroundRemover = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="bg-remove-interface">
            <div class="upload-section">
                <label class="upload-label">
                    <input type="file" id="bg-remove-files" accept="image/*" multiple onchange="loadImagesForBgRemove(this)" style="display: none;">
                    <div class="upload-box">
                        <i class="fas fa-user-slash"></i>
                        <p>Select Images to Remove Background</p>
                        <span>Supports JPG, PNG, WebP - Multiple files allowed</span>
                    </div>
                </label>
            </div>

            <div id="bg-remove-preview" class="preview-section hidden">
                <div class="preview-grid">
                    <div class="preview-item">
                        <h4>Original</h4>
                        <img id="bg-original" alt="Original">
                    </div>
                    <div class="preview-item">
                        <h4>Preview</h4>
                        <canvas id="bg-preview-canvas"></canvas>
                    </div>
                </div>
            </div>

            <div id="bg-remove-options" class="options-section hidden">
                <h3>Background Removal Options</h3>
                
                <div class="form-group">
                    <label>Detection Mode:</label>
                    <select id="bg-detect-mode" onchange="updateBgPreview()">
                        <option value="auto">Auto Detect</option>
                        <option value="person">Person/Portrait</option>
                        <option value="product">Product/Object</option>
                        <option value="edge">Edge Detection</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Edge Smoothing: <span id="smooth-value">5</span></label>
                    <input type="range" id="bg-smooth" min="0" max="20" value="5" oninput="updateBgPreview()">
                </div>

                <div class="form-group">
                    <label>Threshold: <span id="threshold-value">128</span></label>
                    <input type="range" id="bg-threshold" min="0" max="255" value="128" oninput="updateBgPreview()">
                </div>

                <div class="bg-output-options">
                    <h4>Output Options</h4>
                    <label>
                        <input type="radio" name="bg-output" value="transparent" checked>
                        Transparent PNG
                    </label>
                    <label>
                        <input type="radio" name="bg-output" value="white">
                        White Background
                    </label>
                    <label>
                        <input type="radio" name="bg-output" value="custom">
                        Custom Color
                        <input type="color" id="bg-custom-color" value="#ffffff">
                    </label>
                </div>

                <button onclick="processBackgroundRemoval()" class="btn-primary">
                    🎨 Remove Background from All Images
                </button>
            </div>

            <div id="bg-remove-results" class="result-section hidden"></div>
        </div>
    `;
};

function loadImagesForBgRemove(input) {
    bgRemoveImages = Array.from(input.files);
    if (!bgRemoveImages.length) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        document.getElementById('bg-original').src = e.target.result;
        document.getElementById('bg-remove-preview').classList.remove('hidden');
        document.getElementById('bg-remove-options').classList.remove('hidden');
        
        // Simple preview simulation
        setTimeout(() => updateBgPreview(), 100);
    };
    reader.readAsDataURL(bgRemoveImages[0]);
}

function updateBgPreview() {
    const smooth = document.getElementById('bg-smooth').value;
    const threshold = document.getElementById('bg-threshold').value;
    
    document.getElementById('smooth-value').textContent = smooth;
    document.getElementById('threshold-value').textContent = threshold;

    // Canvas preview (simplified - actual processing on backend)
    const canvas = document.getElementById('bg-preview-canvas');
    const ctx = canvas.getContext('2d');
    const img = document.getElementById('bg-original');
    
    canvas.width = img.naturalWidth || 400;
    canvas.height = img.naturalHeight || 400;
    
    ctx.drawImage(img, 0, 0);
    ctx.globalAlpha = 0.7;
    ctx.fillStyle = 'rgba(0, 255, 0, 0.3)';
    ctx.fillRect(0, 0, canvas.width, canvas.height);
}

async function processBackgroundRemoval() {
    if (!bgRemoveImages.length) return;

    const mode = document.getElementById('bg-detect-mode').value;
    const smooth = document.getElementById('bg-smooth').value;
    const threshold = document.getElementById('bg-threshold').value;
    const outputType = document.querySelector('input[name="bg-output"]:checked').value;
    const customColor = document.getElementById('bg-custom-color').value;

    const formData = new FormData();
    bgRemoveImages.forEach(file => formData.append('images', file));
    formData.append('mode', mode);
    formData.append('smooth', smooth);
    formData.append('threshold', threshold);
    formData.append('outputType', outputType);
    formData.append('customColor', customColor);

    document.getElementById('bg-remove-results').innerHTML = '<div class="processing">🎨 Removing backgrounds... This may take a moment...</div>';
    document.getElementById('bg-remove-results').classList.remove('hidden');

    try {
        const response = await fetch('/api/image/remove-background', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayBgRemoveResults(data.images);
        } else {
            document.getElementById('bg-remove-results').innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('bg-remove-results').innerHTML = `<div class="error">Error removing background: ${error.message}</div>`;
    }
}

function displayBgRemoveResults(images) {
    let html = '<div class="result-success"><h3>✅ Backgrounds Removed Successfully!</h3><div class="bg-results-grid">';
    
    images.forEach(img => {
        html += `
            <div class="bg-result-item">
                <div class="bg-comparison">
                    <img src="${img.url}" alt="${img.filename}" class="transparent-bg">
                </div>
                <div class="bg-result-info">
                    <p class="filename">${img.filename}</p>
                    <p class="file-size">${(img.size / 1024).toFixed(2)} KB</p>
                    <button onclick="window.open('${img.url}', '_blank')" class="download-btn-small">
                        📥 Download
                    </button>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    
    if (images.length > 1) {
        html += '<button onclick="downloadAllBgRemoved()" class="download-btn">📥 Download All as ZIP</button>';
    }
    
    html += '</div>';
    document.getElementById('bg-remove-results').innerHTML = html;
}

function downloadAllBgRemoved() {
    alert('ZIP download functionality will be added soon!');
}
