// Background Changer / Replacer

let bgChangeImage = null;
let bgReplaceImage = null;

window.initializeBackgroundChanger = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="bg-change-interface">
            <div class="upload-row">
                <div class="upload-section">
                    <h3>Main Image</h3>
                    <label class="upload-label">
                        <input type="file" accept="image/*" onchange="loadMainImageForBg(this)" style="display: none;">
                        <div class="upload-box">
                            <i class="fas fa-image"></i>
                            <p>Select Main Image</p>
                        </div>
                    </label>
                </div>
                <div class="upload-section">
                    <h3>New Background</h3>
                    <label class="upload-label">
                        <input type="file" accept="image/*" onchange="loadBackgroundImage(this)" style="display: none;">
                        <div class="upload-box">
                            <i class="fas fa-mountain"></i>
                            <p>Select Background</p>
                            <span>Or choose preset below</span>
                        </div>
                    </label>
                </div>
            </div>

            <div id="bg-change-presets" class="presets-section">
                <h3>Background Presets</h3>
                <div class="bg-preset-grid">
                    <div class="bg-preset" onclick="selectBgPreset('gradient-blue')" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                        <span>Gradient Blue</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('gradient-sunset')" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                        <span>Sunset</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('gradient-ocean')" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
                        <span>Ocean</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('gradient-forest')" style="background: linear-gradient(135deg, #0ba360 0%, #3cba92 100%);">
                        <span>Forest</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('solid-white')" style="background: #ffffff; border: 2px solid #ddd;">
                        <span>White</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('solid-black')" style="background: #000000;">
                        <span style="color: white;">Black</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('blur')" style="background: rgba(255,255,255,0.8); backdrop-filter: blur(10px);">
                        <span>Blur Original</span>
                    </div>
                    <div class="bg-preset" onclick="selectBgPreset('custom-color')">
                        <input type="color" id="custom-bg-color" value="#4F46E5">
                        <span>Custom Color</span>
                    </div>
                </div>
                <input type="hidden" id="selected-bg-preset" value="">
            </div>

            <div id="bg-change-preview" class="preview-section hidden">
                <h3>Preview</h3>
                <canvas id="bg-change-canvas"></canvas>
            </div>

            <div id="bg-change-options" class="options-section hidden">
                <h3>Adjustment Options</h3>
                
                <div class="form-group">
                    <label>Subject Position:</label>
                    <div class="position-grid">
                        <button onclick="setSubjectPosition('top-left')">↖</button>
                        <button onclick="setSubjectPosition('top')">↑</button>
                        <button onclick="setSubjectPosition('top-right')">↗</button>
                        <button onclick="setSubjectPosition('left')">←</button>
                        <button onclick="setSubjectPosition('center')" class="active">●</button>
                        <button onclick="setSubjectPosition('right')">→</button>
                        <button onclick="setSubjectPosition('bottom-left')">↙</button>
                        <button onclick="setSubjectPosition('bottom')">↓</button>
                        <button onclick="setSubjectPosition('bottom-right')">↘</button>
                    </div>
                    <input type="hidden" id="subject-position" value="center">
                </div>

                <div class="form-group">
                    <label>Subject Size: <span id="subject-size-value">100</span>%</label>
                    <input type="range" id="subject-size" min="50" max="150" value="100" oninput="updateBgChangePreview()">
                </div>

                <div class="form-group">
                    <label>Edge Feathering: <span id="feather-value">5</span>px</label>
                    <input type="range" id="edge-feather" min="0" max="50" value="5" oninput="updateBgChangePreview()">
                </div>

                <div class="form-group">
                    <label>Background Blur: <span id="bg-blur-value">0</span>px</label>
                    <input type="range" id="bg-blur" min="0" max="20" value="0" oninput="updateBgChangePreview()">
                </div>

                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="bg-output-format">
                        <option value="png">PNG (Transparent Support)</option>
                        <option value="jpg">JPG (Smaller Size)</option>
                        <option value="webp">WebP (Best Compression)</option>
                    </select>
                </div>

                <button onclick="changeBackground()" class="btn-primary">
                    🎨 Change Background
                </button>
            </div>

            <div id="bg-change-result" class="result-section hidden"></div>
        </div>
    `;
};

function loadMainImageForBg(input) {
    bgChangeImage = input.files[0];
    if (!bgChangeImage) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        document.getElementById('bg-change-preview').classList.remove('hidden');
        document.getElementById('bg-change-options').classList.remove('hidden');
        updateBgChangePreview();
    };
    reader.readAsDataURL(bgChangeImage);
}

function loadBackgroundImage(input) {
    bgReplaceImage = input.files[0];
    if (!bgReplaceImage) return;
    
    document.getElementById('selected-bg-preset').value = 'custom-image';
    updateBgChangePreview();
}

function selectBgPreset(preset) {
    document.querySelectorAll('.bg-preset').forEach(el => el.classList.remove('active'));
    event.currentTarget.classList.add('active');
    document.getElementById('selected-bg-preset').value = preset;
    updateBgChangePreview();
}

function setSubjectPosition(position) {
    document.querySelectorAll('.position-grid button').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    document.getElementById('subject-position').value = position;
    updateBgChangePreview();
}

function updateBgChangePreview() {
    const size = document.getElementById('subject-size').value;
    const feather = document.getElementById('edge-feather').value;
    const blur = document.getElementById('bg-blur').value;
    
    document.getElementById('subject-size-value').textContent = size;
    document.getElementById('feather-value').textContent = feather;
    document.getElementById('bg-blur-value').textContent = blur;

    // Simple canvas preview
    const canvas = document.getElementById('bg-change-canvas');
    const ctx = canvas.getContext('2d');
    canvas.width = 600;
    canvas.height = 400;
    
    // Show preview placeholder
    ctx.fillStyle = '#f3f4f6';
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    ctx.fillStyle = '#6b7280';
    ctx.font = '16px Arial';
    ctx.textAlign = 'center';
    ctx.fillText('Preview will appear here after processing', canvas.width/2, canvas.height/2);
}

async function changeBackground() {
    if (!bgChangeImage) {
        alert('Please select a main image first');
        return;
    }

    const formData = new FormData();
    formData.append('mainImage', bgChangeImage);
    
    const preset = document.getElementById('selected-bg-preset').value;
    if (bgReplaceImage && preset === 'custom-image') {
        formData.append('bgImage', bgReplaceImage);
    } else if (preset === 'custom-color') {
        formData.append('bgColor', document.getElementById('custom-bg-color').value);
    } else if (preset) {
        formData.append('preset', preset);
    }

    formData.append('position', document.getElementById('subject-position').value);
    formData.append('size', document.getElementById('subject-size').value);
    formData.append('feather', document.getElementById('edge-feather').value);
    formData.append('blur', document.getElementById('bg-blur').value);
    formData.append('format', document.getElementById('bg-output-format').value);

    document.getElementById('bg-change-result').innerHTML = '<div class="processing">🎨 Changing background...</div>';
    document.getElementById('bg-change-result').classList.remove('hidden');

    try {
        const response = await fetch('/api/image/change-background', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayBgChangeResult(data);
        } else {
            document.getElementById('bg-change-result').innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('bg-change-result').innerHTML = `<div class="error">Error changing background: ${error.message}</div>`;
    }
}

function displayBgChangeResult(data) {
    const html = `
        <div class="result-success">
            <h3>✅ Background Changed Successfully!</h3>
            <div class="bg-change-comparison">
                <img src="${data.url}" alt="Result" class="transparent-bg">
            </div>
            <div class="file-info">
                <p><strong>Size:</strong> ${(data.size / 1024).toFixed(2)} KB</p>
                <p><strong>Dimensions:</strong> ${data.width}x${data.height}</p>
            </div>
            <button onclick="window.open('${data.url}', '_blank')" class="download-btn">
                📥 Download Result
            </button>
        </div>
    `;
    document.getElementById('bg-change-result').innerHTML = html;
}
