// Image Background Blur Tool

let blurBgImages = [];

window.initializeBackgroundBlur = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="bg-blur-interface">
            <div class="upload-section">
                <label class="upload-label">
                    <input type="file" id="blur-bg-files" accept="image/*" multiple onchange="loadImagesForBlur(this)" style="display: none;">
                    <div class="upload-box">
                        <i class="fas fa-adjust"></i>
                        <p>Select Images to Blur Background</p>
                        <span>Portrait mode effect - blur background, keep subject sharp</span>
                    </div>
                </label>
            </div>

            <div id="blur-preview" class="preview-section hidden">
                <div class="preview-comparison">
                    <div class="preview-item">
                        <h4>Original</h4>
                        <img id="blur-original" alt="Original">
                    </div>
                    <div class="preview-item">
                        <h4>Preview (Simulated)</h4>
                        <div id="blur-preview-container">
                            <img id="blur-preview-img" alt="Preview">
                        </div>
                    </div>
                </div>
            </div>

            <div id="blur-options" class="options-section hidden">
                <h3>Blur Settings</h3>
                
                <div class="form-group">
                    <label>Blur Intensity: <span id="blur-intensity-value">10</span></label>
                    <input type="range" id="blur-intensity" min="1" max="50" value="10" oninput="updateBlurPreview()">
                </div>

                <div class="form-group">
                    <label>Subject Detection:</label>
                    <select id="subject-detect" onchange="updateBlurPreview()">
                        <option value="auto">Auto Detect Subject</option>
                        <option value="center">Center Focus</option>
                        <option value="face">Face Detection</option>
                        <option value="edge">Edge Detection</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Focus Area Size: <span id="focus-size-value">50</span>%</label>
                    <input type="range" id="focus-size" min="10" max="90" value="50" oninput="updateBlurPreview()">
                </div>

                <div class="form-group">
                    <label>Transition Smoothness: <span id="transition-value">20</span></label>
                    <input type="range" id="blur-transition" min="0" max="50" value="20" oninput="updateBlurPreview()">
                </div>

                <div class="blur-presets">
                    <h4>Quick Presets</h4>
                    <button class="preset-btn" onclick="applyBlurPreset('subtle')">Subtle</button>
                    <button class="preset-btn active" onclick="applyBlurPreset('medium')">Medium</button>
                    <button class="preset-btn" onclick="applyBlurPreset('strong')">Strong</button>
                    <button class="preset-btn" onclick="applyBlurPreset('extreme')">Extreme</button>
                </div>

                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="blur-output-format">
                        <option value="jpg">JPG</option>
                        <option value="png">PNG</option>
                        <option value="webp">WebP</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Quality:</label>
                    <input type="range" id="blur-quality" min="1" max="100" value="90">
                    <span id="blur-quality-value">90</span>%
                </div>

                <button onclick="applyBackgroundBlur()" class="btn-primary">
                    🎨 Apply Background Blur to All
                </button>
            </div>

            <div id="blur-results" class="result-section hidden"></div>
        </div>
    `;
};

function loadImagesForBlur(input) {
    blurBgImages = Array.from(input.files);
    if (!blurBgImages.length) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        const img = document.getElementById('blur-original');
        img.src = e.target.result;
        
        const previewImg = document.getElementById('blur-preview-img');
        previewImg.src = e.target.result;
        
        document.getElementById('blur-preview').classList.remove('hidden');
        document.getElementById('blur-options').classList.remove('hidden');
        
        updateBlurPreview();
    };
    reader.readAsDataURL(blurBgImages[0]);
}

function updateBlurPreview() {
    const intensity = document.getElementById('blur-intensity').value;
    const focusSize = document.getElementById('focus-size').value;
    const transition = document.getElementById('blur-transition').value;
    const quality = document.getElementById('blur-quality').value;
    
    document.getElementById('blur-intensity-value').textContent = intensity;
    document.getElementById('focus-size-value').textContent = focusSize;
    document.getElementById('transition-value').textContent = transition;
    document.getElementById('blur-quality-value').textContent = quality;

    // Apply CSS blur preview (simplified)
    const previewImg = document.getElementById('blur-preview-img');
    previewImg.style.filter = `blur(${intensity/5}px)`;
    
    // Create radial gradient mask effect
    const container = document.getElementById('blur-preview-container');
    const maskSize = focusSize;
    container.style.background = `radial-gradient(circle at center, transparent ${maskSize}%, rgba(0,0,0,0.3) 100%)`;
}

function applyBlurPreset(preset) {
    document.querySelectorAll('.preset-btn').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');

    const presets = {
        subtle: { intensity: 5, focus: 60, transition: 30 },
        medium: { intensity: 10, focus: 50, transition: 20 },
        strong: { intensity: 20, focus: 40, transition: 15 },
        extreme: { intensity: 40, focus: 30, transition: 10 }
    };

    const settings = presets[preset];
    document.getElementById('blur-intensity').value = settings.intensity;
    document.getElementById('focus-size').value = settings.focus;
    document.getElementById('blur-transition').value = settings.transition;
    
    updateBlurPreview();
}

async function applyBackgroundBlur() {
    if (!blurBgImages.length) return;

    const formData = new FormData();
    blurBgImages.forEach(file => formData.append('images', file));
    
    formData.append('intensity', document.getElementById('blur-intensity').value);
    formData.append('detection', document.getElementById('subject-detect').value);
    formData.append('focusSize', document.getElementById('focus-size').value);
    formData.append('transition', document.getElementById('blur-transition').value);
    formData.append('format', document.getElementById('blur-output-format').value);
    formData.append('quality', document.getElementById('blur-quality').value);

    document.getElementById('blur-results').innerHTML = '<div class="processing">🎨 Applying background blur...</div>';
    document.getElementById('blur-results').classList.remove('hidden');

    try {
        const response = await fetch('/api/image/blur-background', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayBlurResults(data.images);
        } else {
            document.getElementById('blur-results').innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('blur-results').innerHTML = `<div class="error">Error applying blur: ${error.message}</div>`;
    }
}

function displayBlurResults(images) {
    let html = '<div class="result-success"><h3>✅ Background Blur Applied!</h3><div class="blur-results-grid">';
    
    images.forEach(img => {
        html += `
            <div class="blur-result-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="blur-result-info">
                    <p class="filename">${img.filename}</p>
                    <p class="file-size">${(img.size / 1024).toFixed(2)} KB</p>
                    <button onclick="window.open('${img.url}', '_blank')" class="download-btn-small">
                        📥 Download
                    </button>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    
    if (images.length > 1) {
        html += '<button onclick="downloadAllBlurred()" class="download-btn">📥 Download All as ZIP</button>';
    }
    
    html += '</div>';
    document.getElementById('blur-results').innerHTML = html;
}

function downloadAllBlurred() {
    alert('ZIP download functionality will be added soon!');
}
