// Audio Visualizer

let audioContext = null;
let analyser = null;
let audioSource = null;
let animationId = null;

window.initializeAudioVisualizer = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-music"></i> Audio Visualizer</h2>
                <p>Visualize audio waveforms and frequency spectrum</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('audio-viz-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop audio file here or click to upload</h3>
                <p>Supports: MP3, WAV, OGG, M4A</p>
                <input type="file" id="audio-viz-input" accept="audio/*" style="display: none;">
            </div>

            <div id="audio-player-section" class="hidden">
                <audio id="audio-player" controls style="width: 100%; margin: 20px 0;"></audio>

                <div class="visualizer-controls">
                    <div class="form-group">
                        <label>Visualization Type:</label>
                        <select id="viz-type">
                            <option value="bars">Frequency Bars</option>
                            <option value="wave">Waveform</option>
                            <option value="circular">Circular</option>
                            <option value="particles">Particles</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Color Scheme:</label>
                        <select id="viz-color">
                            <option value="rainbow">Rainbow</option>
                            <option value="fire">Fire</option>
                            <option value="ocean">Ocean</option>
                            <option value="neon">Neon</option>
                            <option value="monochrome">Monochrome</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Smoothing:</label>
                        <input type="range" id="viz-smoothing" min="0" max="1" step="0.1" value="0.8">
                    </div>
                </div>

                <div class="visualizer-canvas">
                    <canvas id="viz-canvas" width="800" height="400"></canvas>
                </div>

                <div class="button-group">
                    <button onclick="startVisualization()" class="btn-primary">
                        <i class="fas fa-play"></i> Start Visualization
                    </button>
                    <button onclick="stopVisualization()" class="btn-secondary">
                        <i class="fas fa-stop"></i> Stop
                    </button>
                    <button onclick="captureVisualization()" class="btn-secondary">
                        <i class="fas fa-camera"></i> Capture Frame
                    </button>
                </div>
            </div>
        </div>
    `;

    document.getElementById('audio-viz-input').addEventListener('change', handleAudioUpload);
};

function handleAudioUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    const audioPlayer = document.getElementById('audio-player');
    audioPlayer.src = URL.createObjectURL(file);

    document.getElementById('audio-player-section').classList.remove('hidden');

    // Initialize audio context
    if (!audioContext) {
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
        analyser = audioContext.createAnalyser();
        analyser.fftSize = 2048;
        
        audioSource = audioContext.createMediaElementSource(audioPlayer);
        audioSource.connect(analyser);
        analyser.connect(audioContext.destination);
    }
}

function startVisualization() {
    const smoothing = parseFloat(document.getElementById('viz-smoothing').value);
    analyser.smoothingTimeConstant = smoothing;

    visualize();
}

function visualize() {
    const canvas = document.getElementById('viz-canvas');
    const ctx = canvas.getContext('2d');
    const vizType = document.getElementById('viz-type').value;
    const colorScheme = document.getElementById('viz-color').value;

    const bufferLength = analyser.frequencyBinCount;
    const dataArray = new Uint8Array(bufferLength);

    function draw() {
        animationId = requestAnimationFrame(draw);
        analyser.getByteFrequencyData(dataArray);

        // Clear canvas
        ctx.fillStyle = 'rgba(0, 0, 0, 0.2)';
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        if (vizType === 'bars') {
            drawBars(ctx, dataArray, bufferLength, colorScheme);
        } else if (vizType === 'wave') {
            drawWaveform(ctx, dataArray, bufferLength, colorScheme);
        } else if (vizType === 'circular') {
            drawCircular(ctx, dataArray, bufferLength, colorScheme);
        } else if (vizType === 'particles') {
            drawParticles(ctx, dataArray, bufferLength, colorScheme);
        }
    }

    draw();
}

function drawBars(ctx, dataArray, bufferLength, colorScheme) {
    const barWidth = (ctx.canvas.width / bufferLength) * 2.5;
    let barHeight;
    let x = 0;

    for (let i = 0; i < bufferLength; i++) {
        barHeight = (dataArray[i] / 255) * ctx.canvas.height;

        ctx.fillStyle = getColor(i, bufferLength, colorScheme);
        ctx.fillRect(x, ctx.canvas.height - barHeight, barWidth, barHeight);

        x += barWidth + 1;
    }
}

function drawWaveform(ctx, dataArray, bufferLength, colorScheme) {
    ctx.lineWidth = 2;
    ctx.strokeStyle = getColor(0, 1, colorScheme);
    ctx.beginPath();

    const sliceWidth = ctx.canvas.width / bufferLength;
    let x = 0;

    for (let i = 0; i < bufferLength; i++) {
        const v = dataArray[i] / 128.0;
        const y = (v * ctx.canvas.height) / 2;

        if (i === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }

        x += sliceWidth;
    }

    ctx.stroke();
}

function drawCircular(ctx, dataArray, bufferLength, colorScheme) {
    const centerX = ctx.canvas.width / 2;
    const centerY = ctx.canvas.height / 2;
    const radius = Math.min(centerX, centerY) - 50;

    for (let i = 0; i < bufferLength; i++) {
        const angle = (i / bufferLength) * Math.PI * 2;
        const amplitude = dataArray[i] / 255;
        const length = radius + amplitude * 100;

        const x1 = centerX + Math.cos(angle) * radius;
        const y1 = centerY + Math.sin(angle) * radius;
        const x2 = centerX + Math.cos(angle) * length;
        const y2 = centerY + Math.sin(angle) * length;

        ctx.strokeStyle = getColor(i, bufferLength, colorScheme);
        ctx.lineWidth = 2;
        ctx.beginPath();
        ctx.moveTo(x1, y1);
        ctx.lineTo(x2, y2);
        ctx.stroke();
    }
}

function drawParticles(ctx, dataArray, bufferLength, colorScheme) {
    for (let i = 0; i < bufferLength; i += 4) {
        const x = (i / bufferLength) * ctx.canvas.width;
        const amplitude = dataArray[i] / 255;
        const y = ctx.canvas.height / 2 + (Math.random() - 0.5) * amplitude * 200;
        const size = amplitude * 10;

        ctx.fillStyle = getColor(i, bufferLength, colorScheme);
        ctx.beginPath();
        ctx.arc(x, y, size, 0, Math.PI * 2);
        ctx.fill();
    }
}

function getColor(index, total, scheme) {
    const hue = (index / total) * 360;
    
    switch (scheme) {
        case 'rainbow':
            return `hsl(${hue}, 100%, 50%)`;
        case 'fire':
            return `hsl(${hue * 0.3}, 100%, 50%)`;
        case 'ocean':
            return `hsl(${180 + hue * 0.3}, 70%, 50%)`;
        case 'neon':
            return `hsl(${hue}, 100%, 70%)`;
        case 'monochrome':
            return `hsl(0, 0%, ${(index / total) * 100}%)`;
        default:
            return `hsl(${hue}, 100%, 50%)`;
    }
}

function stopVisualization() {
    if (animationId) {
        cancelAnimationFrame(animationId);
        animationId = null;
    }
}

function captureVisualization() {
    const canvas = document.getElementById('viz-canvas');
    const link = document.createElement('a');
    link.download = 'visualization.png';
    link.href = canvas.toDataURL();
    link.click();
}
