// Audio Extractor from Video

let videoForAudio = null;

window.initializeAudioExtractor = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-music"></i> Audio Extractor</h2>
                <p>Extract audio from video files</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('audio-extract-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop video file here or click to upload</h3>
                <p>Supports: MP4, AVI, MOV, MKV, WebM</p>
                <input type="file" id="audio-extract-input" accept="video/*" style="display: none;">
            </div>

            <div id="audio-extract-preview" class="preview-section hidden">
                <h3>Video Preview</h3>
                <video id="audio-video-player" controls style="max-width: 100%; max-height: 400px;"></video>
                <div id="audio-video-info"></div>
            </div>

            <div id="audio-extract-options" class="options-section hidden">
                <h3>Audio Options</h3>
                
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="audio-format">
                        <option value="mp3">MP3</option>
                        <option value="wav">WAV</option>
                        <option value="aac">AAC</option>
                        <option value="ogg">OGG</option>
                        <option value="flac">FLAC</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Quality:</label>
                    <select id="audio-quality">
                        <option value="high">High (320 kbps)</option>
                        <option value="medium" selected>Medium (192 kbps)</option>
                        <option value="low">Low (128 kbps)</option>
                    </select>
                </div>

                <button onclick="extractAudio()" class="btn-primary">
                    <i class="fas fa-file-audio"></i> Extract Audio
                </button>
            </div>

            <div id="audio-extract-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('audio-extract-input').addEventListener('change', handleAudioExtractUpload);
};

function handleAudioExtractUpload(e) {
    videoForAudio = e.target.files[0];
    if (!videoForAudio) return;

    const player = document.getElementById('audio-video-player');
    player.src = URL.createObjectURL(videoForAudio);

    document.getElementById('audio-video-info').innerHTML = `
        <p><strong>File:</strong> ${videoForAudio.name}</p>
        <p><strong>Size:</strong> ${(videoForAudio.size / 1024 / 1024).toFixed(2)} MB</p>
        <p><strong>Type:</strong> ${videoForAudio.type}</p>
    `;

    document.getElementById('audio-extract-preview').classList.remove('hidden');
    document.getElementById('audio-extract-options').classList.remove('hidden');
}

async function extractAudio() {
    if (!videoForAudio) return;

    const resultsDiv = document.getElementById('audio-extract-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Extracting audio...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('video', videoForAudio);
    formData.append('format', document.getElementById('audio-format').value);
    formData.append('quality', document.getElementById('audio-quality').value);

    try {
        const response = await fetch('/api/video/extract-audio', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Audio Extracted!</h3>
                    <audio controls src="${data.url}" style="width: 100%; margin: 20px 0;"></audio>
                    <p><strong>Format:</strong> ${document.getElementById('audio-format').value.toUpperCase()}</p>
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download Audio
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
