// Tool Search and Filter Functionality

// Category mappings for tools
const toolCategories = {
    // PDF Tools
    'merge-pdf': { categories: ['pdf'], keywords: 'merge combine join unite pdf files' },
    'split-pdf': { categories: ['pdf'], keywords: 'split separate divide extract pdf pages' },
    'organize-pdf': { categories: ['pdf'], keywords: 'organize sort rearrange reorder pdf pages' },
    'rotate-pdf': { categories: ['pdf'], keywords: 'rotate turn flip orientation pdf' },
    'crop-pdf': { categories: ['pdf'], keywords: 'crop trim cut resize pdf pages' },
    'page-numbers': { categories: ['pdf'], keywords: 'page numbers add insert pdf' },
    'pdf-splitter': { categories: ['pdf'], keywords: 'split pdf advanced pages ranges' },
    'pdf-rotator': { categories: ['pdf'], keywords: 'rotate pdf pages orientation' },
    'pdf-compressor': { categories: ['pdf', 'compress'], keywords: 'compress reduce shrink pdf size' },
    'pdf-merger-advanced': { categories: ['pdf'], keywords: 'merge combine pdf advanced' },
    'pdf-form-filler': { categories: ['pdf', 'utility'], keywords: 'pdf form fill fillable fields' },
    'edit-pdf': { categories: ['pdf'], keywords: 'edit modify change pdf text' },
    'pdf-ocr': { categories: ['pdf', 'utility'], keywords: 'ocr text recognition extract pdf' },
    'ocr-pdf': { categories: ['pdf', 'utility'], keywords: 'ocr text recognition pdf scan' },
    
    // Compression Tools
    'compress-pdf': { categories: ['pdf', 'compress'], keywords: 'compress reduce shrink pdf size optimize' },
    'compress-image': { categories: ['image', 'compress'], keywords: 'compress reduce shrink image size optimize' },
    'compress-video': { categories: ['compress', 'utility'], keywords: 'compress reduce shrink video size' },
    'image-optimizer': { categories: ['image', 'compress'], keywords: 'optimize image compress quality size' },
    'resize-pdf': { categories: ['pdf'], keywords: 'resize scale change pdf dimensions' },
    'zip-files': { categories: ['compress', 'utility'], keywords: 'zip create archive compress files folder package' },
    'unzip-files': { categories: ['compress', 'utility'], keywords: 'unzip extract decompress archive files unarchive' },
    
    // Conversion Tools
    'pdf-to-word': { categories: ['convert'], keywords: 'convert pdf word doc docx document' },
    'word-to-pdf': { categories: ['convert'], keywords: 'convert word doc docx pdf document' },
    'pdf-to-image': { categories: ['convert', 'image'], keywords: 'convert pdf image jpg png webp gif bmp all formats' },
    'image-to-pdf': { categories: ['convert', 'image'], keywords: 'convert image to pdf jpg png webp gif heic svg all formats' },
    'pdf-to-jpg': { categories: ['convert'], keywords: 'convert pdf jpg jpeg image picture' },
    'jpg-to-pdf': { categories: ['convert'], keywords: 'convert jpg jpeg image pdf' },
    'image-converter': { categories: ['convert', 'image'], keywords: 'convert image format jpg png webp gif' },
    'jpg-to-png': { categories: ['convert', 'image'], keywords: 'convert jpg jpeg png transparent' },
    'png-to-jpg': { categories: ['convert', 'image'], keywords: 'convert png jpg jpeg image' },
    'webp-to-jpg': { categories: ['convert', 'image'], keywords: 'convert webp jpg jpeg image' },
    'webp-to-png': { categories: ['convert', 'image'], keywords: 'convert webp png transparent' },
    'png-to-webp': { categories: ['convert', 'image'], keywords: 'convert png webp compression' },
    'jpg-to-webp': { categories: ['convert', 'image'], keywords: 'convert jpg jpeg webp compression' },
    'pdf-to-ppt': { categories: ['convert'], keywords: 'convert pdf powerpoint ppt pptx presentation' },
    'ppt-to-pdf': { categories: ['convert'], keywords: 'convert powerpoint ppt pptx pdf presentation' },
    'pdf-to-excel': { categories: ['convert'], keywords: 'convert pdf excel xls xlsx spreadsheet' },
    'excel-to-pdf': { categories: ['convert'], keywords: 'convert excel xls xlsx pdf spreadsheet' },
    'html-to-pdf': { categories: ['convert'], keywords: 'convert html web pdf document' },
    'pdf-to-pdfa': { categories: ['convert'], keywords: 'convert pdf pdfa archive standard' },
    'batch-converter': { categories: ['convert', 'image'], keywords: 'batch convert multiple images format' },
    'video-to-gif': { categories: ['convert', 'creative'], keywords: 'convert video gif animation' },
    
    // Image Tools
    'image-resizer': { categories: ['image'], keywords: 'resize scale dimensions image photos' },
    'image-cropper': { categories: ['image'], keywords: 'crop cut trim image photos' },
    'watermark-tool': { categories: ['image', 'security'], keywords: 'watermark add logo brand image' },
    'watermark': { categories: ['pdf', 'security'], keywords: 'watermark add pdf logo brand' },
    'image-filters': { categories: ['image', 'creative'], keywords: 'filters effects image enhance' },
    'image-effects': { categories: ['image', 'creative'], keywords: 'effects filters image enhance modify' },
    'image-comparison': { categories: ['image', 'utility'], keywords: 'compare diff difference image side by side' },
    'grayscale-converter': { categories: ['image', 'creative'], keywords: 'grayscale black white monochrome bw' },
    'sepia-filter': { categories: ['image', 'creative'], keywords: 'sepia vintage old photo retro' },
    'image-rotate': { categories: ['image'], keywords: 'rotate turn angle image orientation' },
    'image-flip': { categories: ['image'], keywords: 'flip mirror horizontal vertical image' },
    
    // Background Tools
    'remove-background': { categories: ['image', 'background'], keywords: 'remove background transparent image' },
    'background-remover': { categories: ['image', 'background'], keywords: 'remove background transparent image photos' },
    'background-changer': { categories: ['image', 'background'], keywords: 'change replace background image' },
    'background-blur': { categories: ['image', 'background'], keywords: 'blur background image bokeh depth' },
    
    // Creative Tools
    'qr-generator': { categories: ['creative', 'utility'], keywords: 'qr code generator create barcode' },
    'image-base64': { categories: ['utility', 'convert'], keywords: 'base64 encode decode image data' },
    'favicon-generator': { categories: ['creative', 'utility'], keywords: 'favicon icon website create' },
    'color-extractor': { categories: ['creative', 'utility'], keywords: 'color palette extract image picker' },
    'bulk-rename': { categories: ['utility'], keywords: 'bulk rename files batch multiple' },
    'collage-maker': { categories: ['creative', 'image'], keywords: 'collage grid photo combine multiple' },
    'meme-generator': { categories: ['creative'], keywords: 'meme create funny image text' },
        // Utility Tools
    'word-counter': { categories: ['text', 'utility'], keywords: 'word counter character counter text analysis count words' },
    'space-counter': { categories: ['text', 'utility'], keywords: 'space counter whitespace count tabs newlines' },
    'typing-speed-master': { categories: ['text', 'utility'], keywords: 'typing speed test wpm words per minute typing master' },
    'screenshot-tool': { categories: ['utility'], keywords: 'screenshot capture screen snap' },
    'screenshot-tool-advanced': { categories: ['utility'], keywords: 'screenshot capture screen advanced' },
    
    // Security Tools
    'protect-pdf': { categories: ['pdf', 'security'], keywords: 'protect password encrypt secure pdf' },
    'unlock-pdf': { categories: ['pdf', 'security'], keywords: 'unlock remove password decrypt pdf' },
    'sign-pdf': { categories: ['pdf', 'security'], keywords: 'sign signature digital pdf esign' },
    'redact-pdf': { categories: ['pdf', 'security'], keywords: 'redact hide remove sensitive pdf information' },
    
    // Utility Tools
    'url-shortener': { categories: ['utility'], keywords: 'url shortener link shorten short' },
    'text-extractor': { categories: ['utility'], keywords: 'extract text ocr pdf image' },
    'resume-builder': { categories: ['utility'], keywords: 'resume cv builder create job application' },
    'compare-pdf': { categories: ['pdf', 'utility'], keywords: 'compare diff difference pdf documents' },
    'repair-pdf': { categories: ['pdf', 'utility'], keywords: 'repair fix corrupted pdf recover' },
    'scan-pdf': { categories: ['pdf', 'utility'], keywords: 'scan image pdf create document' }
};

// Initialize search and filter
function initSearchFilter() {
    updateToolCount();
    
    // Add category and keyword attributes to all tool cards
    document.querySelectorAll('.tool-card').forEach(card => {
        const toolName = card.getAttribute('data-tool');
        const toolInfo = toolCategories[toolName];
        
        if (toolInfo) {
            card.setAttribute('data-category', toolInfo.categories.join(' '));
            card.setAttribute('data-keywords', toolInfo.keywords);
        }
    });
}

// Search tools
function searchTools() {
    const searchTerm = document.getElementById('tool-search').value.toLowerCase();
    const toolCards = document.querySelectorAll('.tool-card');
    let visibleCount = 0;
    
    toolCards.forEach(card => {
        const toolName = card.querySelector('h3').textContent.toLowerCase();
        const toolDesc = card.querySelector('p').textContent.toLowerCase();
        const keywords = card.getAttribute('data-keywords') || '';
        
        const matches = toolName.includes(searchTerm) || 
                       toolDesc.includes(searchTerm) || 
                       keywords.includes(searchTerm);
        
        if (matches || searchTerm === '') {
            card.style.display = 'flex';
            visibleCount++;
        } else {
            card.style.display = 'none';
        }
    });
    
    updateToolCount(visibleCount);
    updateSectionVisibility();
    
    // Show "no results" message if needed
    showNoResultsMessage(visibleCount === 0 && searchTerm !== '');
}

// Filter by category
function filterByCategory(category) {
    const toolCards = document.querySelectorAll('.tool-card');
    const categoryBtns = document.querySelectorAll('.category-btn');
    let visibleCount = 0;
    
    // Update active button
    categoryBtns.forEach(btn => btn.classList.remove('active'));
    event.target.closest('.category-btn').classList.add('active');
    
    // Clear search
    const searchInput = document.getElementById('tool-search');
    if (searchInput) searchInput.value = '';
    
    // Filter tools
    toolCards.forEach(card => {
        const categories = card.getAttribute('data-category') || '';
        
        if (category === 'all' || categories.includes(category)) {
            card.style.display = 'flex';
            visibleCount++;
        } else {
            card.style.display = 'none';
        }
    });
    
    updateToolCount(visibleCount);
    updateSectionVisibility();
    showNoResultsMessage(visibleCount === 0);
}

// Update section visibility based on visible tools
function updateSectionVisibility() {
    const sections = document.querySelectorAll('.section');
    
    sections.forEach(section => {
        const toolCards = section.querySelectorAll('.tool-card');
        let hasVisibleTools = false;
        
        toolCards.forEach(card => {
            if (card.style.display !== 'none') {
                hasVisibleTools = true;
            }
        });
        
        // Hide section if no visible tools
        if (hasVisibleTools) {
            section.style.display = 'block';
        } else {
            section.style.display = 'none';
        }
    });
}

// Update tool count
function updateToolCount(visibleCount = null) {
    const totalTools = document.querySelectorAll('.tool-card').length;
    const visible = visibleCount !== null ? visibleCount : totalTools;
    
    const visibleEl = document.getElementById('visible-tools-count');
    const totalEl = document.getElementById('total-tools-count');
    
    if (visibleEl) visibleEl.textContent = visible;
    if (totalEl) totalEl.textContent = totalTools;
}

// Show no results message
function showNoResultsMessage(show) {
    let noResultsDiv = document.getElementById('no-results-message');
    
    if (show) {
        if (!noResultsDiv) {
            noResultsDiv = document.createElement('div');
            noResultsDiv.id = 'no-results-message';
            noResultsDiv.className = 'no-results';
            noResultsDiv.innerHTML = `
                <i class="fas fa-search"></i>
                <h3>No Tools Found</h3>
                <p>Try adjusting your search or filter criteria</p>
            `;
            document.getElementById('main-content').appendChild(noResultsDiv);
        }
        noResultsDiv.style.display = 'block';
    } else {
        if (noResultsDiv) {
            noResultsDiv.style.display = 'none';
        }
    }
}

// Initialize when DOM is loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initSearchFilter);
} else {
    initSearchFilter();
}

// Make tool cards clickable
document.addEventListener('DOMContentLoaded', function() {
    const toolCards = document.querySelectorAll('.tool-card');
    
    toolCards.forEach(card => {
        card.addEventListener('click', function(e) {
            // Don't trigger if clicking on a button
            if (e.target.tagName === 'BUTTON' || e.target.closest('button')) {
                return;
            }
            
            const toolId = this.getAttribute('data-tool');
            if (toolId && typeof openToolPage === 'function') {
                openToolPage(toolId);
            }
        });
    });
});
