// Mobile navbar behavior: hamburger toggle and submenu accordion
document.addEventListener('DOMContentLoaded', function () {
  const hamburger = document.querySelector('.hamburger-menu');
  const nav = document.querySelector('.nav');

  // Create overlay
  const overlay = document.createElement('div');
  overlay.className = 'nav-overlay';
  document.body.appendChild(overlay);

  // Prevent clicks on nav from closing the menu
  nav.addEventListener('click', (e) => e.stopPropagation());

  if (hamburger && nav) {
    hamburger.addEventListener('click', () => {
      nav.classList.toggle('show');
      hamburger.classList.toggle('active');
      overlay.style.display = nav.classList.contains('show') ? 'block' : 'none';
      // Ensure nav links don't propagate clicks that may close the menu
      attachNavLinkHandlers();
    });
  }

  // Close menu when clicking overlay
  overlay.addEventListener('click', () => {
    nav.classList.remove('show');
    hamburger.classList.remove('active');
    overlay.style.display = 'none';
  });

  // Convert desktop hover dropdowns to click-to-open accordion on mobile
  function setupMobileSubmenus() {
    const wrappers = document.querySelectorAll('.nav-item-wrapper');
    wrappers.forEach(wrapper => {
      // Ensure we only attach for mobile widths
      const toggle = wrapper.querySelector('.nav-item');
      const submenu = wrapper.querySelector('.dropdown-menu');
      if (!submenu) return;

      // Add an expand icon if not present
      if (!wrapper.querySelector('.submenu-toggle')){
        const exp = document.createElement('button');
        exp.type = 'button';
        exp.className = 'submenu-toggle';
        exp.setAttribute('aria-expanded','false');
        exp.innerHTML = '<i class="fas fa-chevron-down"></i>';
        toggle.appendChild(exp);

        exp.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();
          const isOpen = wrapper.classList.contains('open');
          if (!isOpen) {
            wrapper.classList.add('open');
            exp.setAttribute('aria-expanded', 'true');
            submenu.style.maxHeight = submenu.scrollHeight + 'px';
          }
          // Submenu stays open once expanded
        });
      }
    });
  }

  // Prevent nav link clicks from bubbling to global handlers; handle SPA navigation here
  function attachNavLinkHandlers() {
    if (!nav) return;
    nav.querySelectorAll('a').forEach(link => {
      // avoid adding duplicate listeners
      if (link.__navHandlerAttached) return;
      const handler = (e) => {
        e.stopPropagation();
        const route = link.getAttribute('data-route') || link.getAttribute('href');
        if (route && route.startsWith('/')) {
          e.preventDefault();
          if (window.router && typeof window.router.navigate === 'function') {
            window.router.navigate(route);
          } else {
            window.location.href = route;
          }
        }
      };
      link.addEventListener('click', handler);
      link.__navHandlerAttached = true;
    });
  }

  // Initialize and on resize
  function onResize() {
    if (window.innerWidth <= 1023) {
      setupMobileSubmenus();
      // Add close button if not present
      if (!nav.querySelector('.nav-close')) {
        const closeBtn = document.createElement('button');
        closeBtn.className = 'nav-close';
        closeBtn.innerHTML = '<i class="fas fa-times"></i>';
        closeBtn.setAttribute('aria-label', 'Close menu');
        nav.appendChild(closeBtn);
        closeBtn.addEventListener('click', () => {
          nav.classList.remove('show');
          hamburger.classList.remove('active');
          overlay.style.display = 'none';
        });
      }
    } else {
      // cleanup mobile inline styles
      document.querySelectorAll('.dropdown-menu').forEach(d => {
        d.style.maxHeight = null;
      });
      document.querySelectorAll('.nav-item-wrapper.open').forEach(w => w.classList.remove('open'));
      // Hide mobile menu and overlay on desktop
      nav.classList.remove('show');
      hamburger.classList.remove('active');
      overlay.style.display = 'none';
      // Remove close button
      const closeBtn = nav.querySelector('.nav-close');
      if (closeBtn) closeBtn.remove();
    }
  }
  // Track last click location to detect internal clicks
  let lastClickInsideNav = false;
  document.addEventListener('click', (e) => {
    lastClickInsideNav = nav && nav.contains(e.target);
    // reset after short delay
    setTimeout(() => { lastClickInsideNav = false; }, 300);
  }, true);

  // Observe class changes on nav; if 'show' is removed immediately after an inside click,
  // assume an external handler mistakenly closed it and restore it.
  if (window.MutationObserver && nav) {
    const mo = new MutationObserver((mutations) => {
      mutations.forEach(m => {
        if (m.attributeName === 'class') {
          const hasShow = nav.classList.contains('show');
          if (!hasShow && lastClickInsideNav) {
            // restore
            nav.classList.add('show');
            if (hamburger) hamburger.classList.add('active');
            overlay.style.display = 'block';
          }
        }
      });
    });
    mo.observe(nav, { attributes: true, attributeFilter: ['class'] });
  }

  window.addEventListener('resize', onResize);
  onResize();
});
