// Navbar Search Functionality with Fuzzy Matching

// All tools data with display names
const allTools = [
    // PDF Tools
    { id: 'merge-pdf', name: 'Merge PDF', category: 'pdf', keywords: 'merge combine join unite pdf files' },
    { id: 'split-pdf', name: 'Split PDF', category: 'pdf', keywords: 'split separate divide extract pdf pages' },
    { id: 'organize-pdf', name: 'Organize PDF', category: 'pdf', keywords: 'organize sort rearrange reorder pdf pages' },
    { id: 'rotate-pdf', name: 'Rotate PDF', category: 'pdf', keywords: 'rotate turn flip orientation pdf' },
    { id: 'crop-pdf', name: 'Crop PDF', category: 'pdf', keywords: 'crop trim cut resize pdf pages' },
    { id: 'page-numbers', name: 'Add Page Numbers', category: 'pdf', keywords: 'page numbers add insert pdf' },
    { id: 'pdf-splitter', name: 'PDF Splitter', category: 'pdf', keywords: 'split pdf advanced pages ranges' },
    { id: 'pdf-rotator', name: 'PDF Rotator', category: 'pdf', keywords: 'rotate pdf pages orientation' },
    { id: 'pdf-compressor', name: 'Compress PDF', category: 'compress', keywords: 'compress reduce shrink pdf size' },
    { id: 'pdf-merger-advanced', name: 'Merge PDF Advanced', category: 'pdf', keywords: 'merge combine pdf advanced' },
    { id: 'pdf-form-filler', name: 'PDF Form Filler', category: 'utility', keywords: 'pdf form fill fillable fields' },
    { id: 'edit-pdf', name: 'Edit PDF', category: 'pdf', keywords: 'edit modify change pdf text' },
    { id: 'pdf-ocr', name: 'PDF OCR', category: 'utility', keywords: 'ocr text recognition extract pdf' },
    { id: 'ocr-pdf', name: 'OCR PDF', category: 'utility', keywords: 'ocr text recognition pdf scan' },
    
    // Compression Tools
    { id: 'compress-pdf', name: 'Compress PDF', category: 'compress', keywords: 'compress reduce shrink pdf size optimize' },
    { id: 'compress-image', name: 'Compress Image', category: 'compress', keywords: 'compress reduce shrink image size optimize' },
    { id: 'compress-video', name: 'Compress Video', category: 'compress', keywords: 'compress reduce shrink video size' },
    { id: 'image-optimizer', name: 'Image Optimizer', category: 'compress', keywords: 'optimize image compress quality size' },
    { id: 'resize-pdf', name: 'Resize PDF', category: 'pdf', keywords: 'resize scale change pdf dimensions' },
    { id: 'zip-files', name: 'Create ZIP', category: 'compress', keywords: 'zip create archive compress files folder package' },
    { id: 'unzip-files', name: 'Extract ZIP', category: 'compress', keywords: 'unzip extract decompress archive files unarchive' },
    
    // Conversion Tools
    { id: 'pdf-to-word', name: 'PDF to Word', category: 'convert', keywords: 'convert pdf word doc docx document' },
    { id: 'word-to-pdf', name: 'Word to PDF', category: 'convert', keywords: 'convert word doc docx pdf document' },
    { id: 'pdf-to-image', name: 'PDF to Image', category: 'convert', keywords: 'convert pdf image jpg png webp gif bmp all formats' },
    { id: 'image-to-pdf', name: 'Image to PDF', category: 'convert', keywords: 'convert image to pdf jpg png webp gif heic svg all formats' },
    { id: 'pdf-to-jpg', name: 'PDF to JPG', category: 'convert', keywords: 'convert pdf jpg jpeg image picture' },
    { id: 'jpg-to-pdf', name: 'JPG to PDF', category: 'convert', keywords: 'convert jpg jpeg image pdf' },
    { id: 'image-converter', name: 'Image Converter', category: 'convert', keywords: 'convert image format jpg png webp gif' },
    { id: 'jpg-to-png', name: 'JPG to PNG', category: 'convert', keywords: 'convert jpg jpeg png transparent' },
    { id: 'png-to-jpg', name: 'PNG to JPG', category: 'convert', keywords: 'convert png jpg jpeg image' },
    { id: 'webp-to-jpg', name: 'WebP to JPG', category: 'convert', keywords: 'convert webp jpg jpeg image' },
    { id: 'webp-to-png', name: 'WebP to PNG', category: 'convert', keywords: 'convert webp png transparent' },
    { id: 'png-to-webp', name: 'PNG to WebP', category: 'convert', keywords: 'convert png webp compression' },
    { id: 'jpg-to-webp', name: 'JPG to WebP', category: 'convert', keywords: 'convert jpg jpeg webp compression' },
    { id: 'pdf-to-ppt', name: 'PDF to PowerPoint', category: 'convert', keywords: 'convert pdf powerpoint ppt pptx presentation' },
    { id: 'ppt-to-pdf', name: 'PowerPoint to PDF', category: 'convert', keywords: 'convert powerpoint ppt pptx pdf presentation' },
    { id: 'pdf-to-excel', name: 'PDF to Excel', category: 'convert', keywords: 'convert pdf excel xls xlsx spreadsheet' },
    { id: 'excel-to-pdf', name: 'Excel to PDF', category: 'convert', keywords: 'convert excel xls xlsx pdf spreadsheet' },
    { id: 'html-to-pdf', name: 'HTML to PDF', category: 'convert', keywords: 'convert html web pdf document' },
    { id: 'pdf-to-pdfa', name: 'PDF to PDF/A', category: 'convert', keywords: 'convert pdf pdfa archive standard' },
    { id: 'batch-converter', name: 'Batch Converter', category: 'convert', keywords: 'batch convert multiple images format' },
    { id: 'video-to-gif', name: 'Video to GIF', category: 'convert', keywords: 'convert video gif animation' },
    { id: 'heic-to-jpg', name: 'HEIC to JPG', category: 'convert', keywords: 'convert heic image jpg' },
    
    // Image Tools
    { id: 'image-resizer', name: 'Image Resizer', category: 'image', keywords: 'resize scale dimensions image photos' },
    { id: 'image-cropper', name: 'Image Cropper', category: 'image', keywords: 'crop cut trim image photos' },
    { id: 'watermark-tool', name: 'Watermark Image', category: 'security', keywords: 'watermark add logo brand image' },
    { id: 'watermark', name: 'PDF Watermark', category: 'security', keywords: 'watermark add pdf logo brand' },
    { id: 'image-filters', name: 'Image Filters', category: 'creative', keywords: 'filters effects image enhance' },
    { id: 'image-effects', name: 'Image Effects', category: 'creative', keywords: 'effects filters image enhance modify' },
    { id: 'image-comparison', name: 'Image Comparison', category: 'utility', keywords: 'compare diff difference image side by side' },
    { id: 'grayscale-converter', name: 'Grayscale Converter', category: 'creative', keywords: 'grayscale black white monochrome bw' },
    { id: 'sepia-filter', name: 'Sepia Filter', category: 'creative', keywords: 'sepia vintage old photo retro' },
    { id: 'image-rotate', name: 'Rotate Image', category: 'image', keywords: 'rotate turn angle image orientation' },
    { id: 'image-flip', name: 'Flip Image', category: 'image', keywords: 'flip mirror horizontal vertical image' },
    { id: 'image-upscaler', name: 'Image Upscaler', category: 'image', keywords: 'upscale enhance enlarge image resolution' },
    { id: 'image-border', name: 'Add Border', category: 'creative', keywords: 'border frame image add' },
    { id: 'image-merger', name: 'Image Merger', category: 'image', keywords: 'merge combine images multiple' },
    { id: 'image-splitter', name: 'Image Splitter', category: 'image', keywords: 'split divide image parts sections' },
    
    // Background Tools
    { id: 'remove-background', name: 'Remove Background', category: 'background', keywords: 'remove background transparent image' },
    { id: 'background-remover', name: 'Background Remover', category: 'background', keywords: 'remove background transparent image photos' },
    { id: 'background-changer', name: 'Change Background', category: 'background', keywords: 'change replace background image' },
    { id: 'background-blur', name: 'Blur Background', category: 'background', keywords: 'blur background image bokeh depth' },
    { id: 'color-extractor', name: 'Color Extractor', category: 'utility', keywords: 'color palette extract image picker' },
    
    // Creative Tools
    { id: 'qr-generator', name: 'QR Code Generator', category: 'creative', keywords: 'qr code generator create barcode' },
    { id: 'qr-code-generator', name: 'QR Code Generator', category: 'creative', keywords: 'qr code generator create barcode' },
    { id: 'image-base64', name: 'Base64 Image', category: 'utility', keywords: 'base64 encode decode image data' },
    { id: 'favicon-generator', name: 'Favicon Generator', category: 'creative', keywords: 'favicon icon website create' },
    { id: 'bulk-rename', name: 'Bulk Rename', category: 'utility', keywords: 'bulk rename files batch multiple' },
    { id: 'collage-maker', name: 'Collage Maker', category: 'creative', keywords: 'collage grid photo combine multiple' },
    { id: 'meme-generator', name: 'Meme Generator', category: 'creative', keywords: 'meme create funny image text' },
    { id: 'gif-maker', name: 'GIF Maker', category: 'creative', keywords: 'gif create animation images' },
    { id: 'gradient-generator', name: 'Gradient Generator', category: 'creative', keywords: 'gradient color background design' },
    { id: 'color-picker', name: 'Color Picker', category: 'creative', keywords: 'color picker eyedropper select color' },
    
    // Text & Utility Tools
    { id: 'word-counter', name: 'Word Counter', category: 'text', keywords: 'word counter character counter text analysis count' },
    { id: 'space-counter', name: 'Space Counter', category: 'text', keywords: 'space counter whitespace count tabs newlines' },
    { id: 'typing-speed-master', name: 'Typing Speed Test', category: 'text', keywords: 'typing speed test wpm words per minute typing master' },
    { id: 'screenshot-tool', name: 'Screenshot Tool', category: 'utility', keywords: 'screenshot capture screen snap' },
    { id: 'case-converter', name: 'Case Converter', category: 'text', keywords: 'case convert uppercase lowercase text' },
    { id: 'text-extractor', name: 'Text Extractor', category: 'utility', keywords: 'extract text ocr pdf image' },
    { id: 'text-to-speech', name: 'Text to Speech', category: 'text', keywords: 'text to speech tts audio voice' },
    { id: 'json-formatter', name: 'JSON Formatter', category: 'utility', keywords: 'json format minify beautify validate' },
    { id: 'csv-to-json', name: 'CSV to JSON', category: 'convert', keywords: 'csv json convert data format' },
    { id: 'json-to-csv', name: 'JSON to CSV', category: 'convert', keywords: 'json csv convert data format' },
    { id: 'xml-formatter', name: 'XML Formatter', category: 'utility', keywords: 'xml format minify beautify validate' },
    { id: 'yaml-to-json', name: 'YAML to JSON', category: 'convert', keywords: 'yaml json convert data' },
    { id: 'html-tool', name: 'HTML Tool', category: 'utility', keywords: 'html format minify beautify validate' },
    { id: 'css-minifier', name: 'CSS Minifier', category: 'utility', keywords: 'css minify minifier compress' },
    { id: 'js-minifier', name: 'JS Minifier', category: 'utility', keywords: 'javascript minify minifier compress' },
    { id: 'base64-converter', name: 'Base64 Converter', category: 'convert', keywords: 'base64 encode decode convert' },
    
    // Security & Document Tools
    { id: 'protect-pdf', name: 'Protect PDF', category: 'security', keywords: 'protect password encrypt secure pdf' },
    { id: 'unlock-pdf', name: 'Unlock PDF', category: 'security', keywords: 'unlock remove password decrypt pdf' },
    { id: 'sign-pdf', name: 'Sign PDF', category: 'security', keywords: 'sign signature digital pdf esign' },
    { id: 'redact-pdf', name: 'Redact PDF', category: 'security', keywords: 'redact hide remove sensitive pdf information' },
    { id: 'pdf-metadata', name: 'PDF Metadata', category: 'utility', keywords: 'metadata pdf remove edit properties' },
    { id: 'pdf-bookmarker', name: 'PDF Bookmarker', category: 'pdf', keywords: 'bookmark bookmark pdf document navigation' },
    { id: 'pdf-page-extractor', name: 'PDF Page Extractor', category: 'pdf', keywords: 'extract page pdf pages ranges' },
    { id: 'compare-pdf', name: 'Compare PDF', category: 'utility', keywords: 'compare diff difference pdf documents' },
    { id: 'repair-pdf', name: 'Repair PDF', category: 'utility', keywords: 'repair fix corrupted pdf recover' },
    
    // More Utility & Generator Tools
    { id: 'resume-builder', name: 'Resume Builder', category: 'utility', keywords: 'resume cv builder create job application' },
    { id: 'password-generator', name: 'Password Generator', category: 'utility', keywords: 'password generate random secure' },
    { id: 'uuid-generator', name: 'UUID Generator', category: 'utility', keywords: 'uuid guid generator create random' },
    { id: 'hash-generator', name: 'Hash Generator', category: 'utility', keywords: 'hash md5 sha sha256 generate' },
    { id: 'random-generator', name: 'Random Generator', category: 'utility', keywords: 'random number generator create' },
    { id: 'url-shortener', name: 'URL Shortener', category: 'utility', keywords: 'url shortener link shorten short' },
    { id: 'qr-url', name: 'URL to QR', category: 'utility', keywords: 'url qr code generate' },
    { id: 'timestamp-converter', name: 'Timestamp Converter', category: 'utility', keywords: 'timestamp unix time convert' },
    { id: 'currency-converter', name: 'Currency Converter', category: 'utility', keywords: 'currency converter exchange rate money' },
    { id: 'unit-converter', name: 'Unit Converter', category: 'utility', keywords: 'unit converter measurement convert' },
    { id: 'age-calculator', name: 'Age Calculator', category: 'utility', keywords: 'age calculator birthday calculate' },
    { id: 'markdown-to-html', name: 'Markdown to HTML', category: 'convert', keywords: 'markdown html convert' },
    { id: 'email-signature', name: 'Email Signature', category: 'utility', keywords: 'email signature generator create' },
    { id: 'business-card', name: 'Business Card', category: 'creative', keywords: 'business card generator create design' },
    { id: 'invoice-generator', name: 'Invoice Generator', category: 'utility', keywords: 'invoice generator create business' },
    { id: 'certificate-generator', name: 'Certificate Generator', category: 'creative', keywords: 'certificate generator create design' },
];

// Fuzzy search algorithm - matches partial words
function fuzzySearch(query, tools) {
    if (!query) return [];
    
    const searchTerms = query.toLowerCase().split(/\s+/).filter(term => term.length > 0);
    
    return tools
        .map(tool => {
            let score = 0;
            const toolText = `${tool.name} ${tool.keywords}`.toLowerCase();
            
            searchTerms.forEach(term => {
                // Exact word match bonus
                if (toolText.includes(` ${term}`) || toolText.startsWith(term)) {
                    score += 100;
                }
                // Substring match
                else if (toolText.includes(term)) {
                    score += 50;
                }
                // Character sequence match (fuzzy)
                else {
                    let tempText = toolText;
                    let fuzzyMatch = true;
                    for (let char of term) {
                        const index = tempText.indexOf(char);
                        if (index === -1) {
                            fuzzyMatch = false;
                            break;
                        }
                        score += 10;
                        tempText = tempText.substring(index + 1);
                    }
                    if (!fuzzyMatch) score = 0;
                }
            });
            
            return { ...tool, score };
        })
        .filter(item => item.score > 0)
        .sort((a, b) => b.score - a.score)
        .slice(0, 8);
}

// Initialize navbar search
function initNavbarSearch() {
    const navbarSearch = document.getElementById('navbar-search');
    if (!navbarSearch) return;
    
    // Create dropdown container
    const dropdownContainer = document.createElement('div');
    dropdownContainer.className = 'navbar-search-dropdown';
    dropdownContainer.id = 'navbar-search-dropdown';
    
    // Insert after search input
    navbarSearch.parentElement.appendChild(dropdownContainer);
    
    // Handle input
    navbarSearch.addEventListener('input', function() {
        const query = this.value;
        const results = fuzzySearch(query, allTools);
        
        // Also filter tool cards on homepage if present
        filterToolCards(query);
        
        if (query.trim() === '') {
            dropdownContainer.style.display = 'none';
            return;
        }
        
        if (results.length === 0) {
            dropdownContainer.innerHTML = '<div class="search-no-results">No tools found</div>';
            dropdownContainer.style.display = 'block';
            return;
        }
        
        dropdownContainer.innerHTML = results
            .map(tool => `<div class="search-result-item" data-tool-id="${tool.id}">${tool.name}</div>`)
            .join('');
        
        // Add click handlers to results
        dropdownContainer.querySelectorAll('.search-result-item').forEach(item => {
            item.addEventListener('click', function(e) {
                const toolId = this.getAttribute('data-tool-id');
                navbarSearch.value = '';
                dropdownContainer.style.display = 'none';
                // Navigate to tool
                if (window.router) {
                    window.router.navigate(`/${toolId}`);
                } else {
                    window.location.href = `/${toolId}`;
                }
            });
        });
        
        dropdownContainer.style.display = 'block';
    });
    
    // Close dropdown when clicking outside
    document.addEventListener('click', function(e) {
        if (!navbarSearch.contains(e.target) && !dropdownContainer.contains(e.target)) {
            dropdownContainer.style.display = 'none';
        }
    });
    
    // Focus to show dropdown again
    navbarSearch.addEventListener('focus', function() {
        if (this.value.trim() !== '') {
            dropdownContainer.style.display = 'block';
        }
    });
}

// Filter tool cards on homepage
function filterToolCards(searchTerm) {
    const toolCards = document.querySelectorAll('.tool-card');
    if (toolCards.length === 0) return; // Not on homepage
    
    searchTerm = searchTerm.toLowerCase();
    let visibleCount = 0;
    
    toolCards.forEach(card => {
        const toolName = card.querySelector('h3')?.textContent.toLowerCase() || '';
        const toolDesc = card.querySelector('p')?.textContent.toLowerCase() || '';
        const keywords = card.getAttribute('data-keywords') || '';
        
        const matches = toolName.includes(searchTerm) || 
                       toolDesc.includes(searchTerm) || 
                       keywords.includes(searchTerm);
        
        if (matches || searchTerm === '') {
            card.style.display = 'flex';
            visibleCount++;
        } else {
            card.style.display = 'none';
        }
    });
    
    // Update tool count if function exists
    if (typeof updateToolCount === 'function') {
        updateToolCount(visibleCount);
    }
    
    // Update section visibility if function exists
    if (typeof updateSectionVisibility === 'function') {
        updateSectionVisibility();
    }
    
    // Show no results message if function exists
    if (typeof showNoResultsMessage === 'function') {
        showNoResultsMessage(visibleCount === 0 && searchTerm !== '');
    }
}

// Helper function to get category icon
function getCategoryIcon(category) {
    const icons = {
        'pdf': 'file-pdf',
        'image': 'image',
        'convert': 'exchange-alt',
        'compress': 'compress',
        'text': 'font',
        'utility': 'tools',
        'security': 'lock',
        'creative': 'palette',
        'background': 'image'
    };
    return icons[category] || 'file';
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initNavbarSearch);
} else {
    initNavbarSearch();
}
