// PDF Viewer Library using PDF.js
class PDFViewer {
    constructor(containerId) {
        this.container = document.getElementById(containerId);
        this.pdfDoc = null;
        this.currentPage = 1;
        this.totalPages = 0;
        this.scale = 1.5;
        this.canvas = null;
        this.ctx = null;
        this.selectedPages = new Set();
    }

    async loadPDF(file) {
        try {
            const arrayBuffer = await file.arrayBuffer();
            const loadingTask = pdfjsLib.getDocument({ data: arrayBuffer });
            this.pdfDoc = await loadingTask.promise;
            this.totalPages = this.pdfDoc.numPages;
            return true;
        } catch (error) {
            console.error('Error loading PDF:', error);
            return false;
        }
    }

    async renderPage(pageNum) {
        if (!this.pdfDoc) return;
        
        const page = await this.pdfDoc.getPage(pageNum);
        const viewport = page.getViewport({ scale: this.scale });
        
        if (!this.canvas) {
            this.canvas = document.createElement('canvas');
            this.ctx = this.canvas.getContext('2d');
        }
        
        this.canvas.width = viewport.width;
        this.canvas.height = viewport.height;
        
        await page.render({
            canvasContext: this.ctx,
            viewport: viewport
        }).promise;
        
        return this.canvas;
    }

    async renderAllPages() {
        const pagesContainer = document.createElement('div');
        pagesContainer.className = 'pdf-pages-grid';
        
        for (let i = 1; i <= this.totalPages; i++) {
            const pageWrapper = document.createElement('div');
            pageWrapper.className = 'pdf-page-wrapper';
            pageWrapper.dataset.pageNum = i;
            
            const canvas = await this.renderPage(i);
            const pageCanvas = canvas.cloneNode();
            const ctx = pageCanvas.getContext('2d');
            ctx.drawImage(canvas, 0, 0);
            
            const pageNumber = document.createElement('div');
            pageNumber.className = 'page-number-badge';
            pageNumber.textContent = `Page ${i}`;
            
            pageWrapper.appendChild(pageCanvas);
            pageWrapper.appendChild(pageNumber);
            pagesContainer.appendChild(pageWrapper);
        }
        
        this.container.innerHTML = '';
        this.container.appendChild(pagesContainer);
        return pagesContainer;
    }

    selectPage(pageNum) {
        if (this.selectedPages.has(pageNum)) {
            this.selectedPages.delete(pageNum);
        } else {
            this.selectedPages.add(pageNum);
        }
    }

    getSelectedPages() {
        return Array.from(this.selectedPages).sort((a, b) => a - b);
    }
}
