// PDF Toolkit Utilities - Fast loading and optimizations

// Lazy load PDF.js and PDF-lib
window.pdfLibsLoaded = false;
window.pdfLibsLoading = false;

async function ensurePDFLibs() {
    if (window.pdfLibsLoaded) return true;
    
    if (window.pdfLibsLoading) {
        await new Promise(resolve => {
            const check = setInterval(() => {
                if (window.pdfLibsLoaded) {
                    clearInterval(check);
                    resolve();
                }
            }, 100);
        });
        return true;
    }

    window.pdfLibsLoading = true;

    try {
        // Check if already loaded
        if (typeof pdfjsLib !== 'undefined' && typeof PDFLib !== 'undefined') {
            window.pdfLibsLoaded = true;
            window.pdfLibsLoading = false;
            return true;
        }

        // Load PDF.js
        await new Promise((resolve, reject) => {
            const pdfjsScript = document.createElement('script');
            pdfjsScript.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js';
            pdfjsScript.onload = resolve;
            pdfjsScript.onerror = reject;
            document.head.appendChild(pdfjsScript);
        });

        // Configure worker
        pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

        // Load PDF-lib
        await new Promise((resolve, reject) => {
            const pdflibScript = document.createElement('script');
            pdflibScript.src = 'https://unpkg.com/pdf-lib@1.17.1/dist/pdf-lib.min.js';
            pdflibScript.onload = resolve;
            pdflibScript.onerror = reject;
            document.head.appendChild(pdflibScript);
        });

        // Make PDF-lib available globally
        window.PDFLib = window['pdf-lib'];
        window.pdfLibsLoaded = true;
        window.pdfLibsLoading = false;
            return true;
        }
        
        return false;
    } catch (error) {
        console.error('Failed to load PDF libraries:', error);
        return false;
    } finally {
        window.pdfLibsLoading = false;
    }
}

// Show loading indicator
function showGlobalLoader(message = 'Processing...') {
    let loader = document.getElementById('global-loader');
    if (!loader) {
        loader = document.createElement('div');
        loader.id = 'global-loader';
        loader.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 99999;
            backdrop-filter: blur(5px);
        `;
        loader.innerHTML = `
            <div style="background: white; padding: 2rem 3rem; border-radius: 12px; text-align: center; box-shadow: 0 20px 60px rgba(0,0,0,0.3);">
                <div class="loader-spinner" style="width: 50px; height: 50px; border: 4px solid var(--bg-tertiary); border-top-color: var(--primary-color); border-radius: 50%; margin: 0 auto 1rem; animation: spin 0.8s linear infinite;"></div>
                <div id="loader-message" style="font-size: 1.1rem; font-weight: 500; color: var(--text-primary);">${message}</div>
            </div>
        `;
        document.body.appendChild(loader);
        
        // Add spin animation if not exists
        if (!document.getElementById('loader-styles')) {
            const style = document.createElement('style');
            style.id = 'loader-styles';
            style.textContent = `
                @keyframes spin {
                    to { transform: rotate(360deg); }
                }
            `;
            document.head.appendChild(style);
        }
    } else {
        loader.style.display = 'flex';
        document.getElementById('loader-message').textContent = message;
    }
}

function hideGlobalLoader() {
    const loader = document.getElementById('global-loader');
    if (loader) {
        loader.style.display = 'none';
    }
}

function updateLoaderMessage(message) {
    const msgEl = document.getElementById('loader-message');
    if (msgEl) msgEl.textContent = message;
}

// File size formatter
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round((bytes / Math.pow(k, i)) * 100) / 100 + ' ' + sizes[i];
}

// Debounce function for performance
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Check if file is valid PDF
async function isValidPDF(file) {
    if (!file) return false;
    if (!file.name.toLowerCase().endsWith('.pdf') && file.type !== 'application/pdf') {
        return false;
    }
    
    try {
        const arrayBuffer = await file.arrayBuffer();
        const uint8Array = new Uint8Array(arrayBuffer);
        const header = String.fromCharCode(...uint8Array.slice(0, 5));
        return header === '%PDF-';
    } catch {
        return false;
    }
}

// Create download link with better UX
function createDownloadLink(blob, filename) {
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.className = 'btn-primary';
    a.style.textDecoration = 'none';
    a.innerHTML = `<i class="fas fa-download"></i> Download ${filename}`;
    
    // Auto-revoke URL after 1 minute to free memory
    setTimeout(() => URL.revokeObjectURL(url), 60000);
    
    return a;
}

// Progress callback for large operations
function createProgressBar() {
    const container = document.createElement('div');
    container.style.cssText = 'width: 100%; background: var(--bg-tertiary); border-radius: 20px; overflow: hidden; height: 8px; margin: 1rem 0;';
    
    const bar = document.createElement('div');
    bar.style.cssText = 'height: 100%; background: linear-gradient(90deg, var(--primary-color), var(--primary-light)); transition: width 0.3s ease; width: 0%;';
    container.appendChild(bar);
    
    return {
        element: container,
        setProgress: (percent) => {
            bar.style.width = Math.min(100, Math.max(0, percent)) + '%';
        }
    };
}

// Optimize PDF for web (linearize)
async function optimizePDFForWeb(pdfDoc) {
    try {
        const pdfBytes = await pdfDoc.save({
            useObjectStreams: true,
            addDefaultPage: false,
            objectsPerTick: 50
        });
        return pdfBytes;
    } catch (error) {
        console.warn('Optimization failed, using standard save:', error);
        return await pdfDoc.save();
    }
}

// Export utilities
window.PDFUtils = {
    ensurePDFLibs,
    showGlobalLoader,
    hideGlobalLoader,
    updateLoaderMessage,
    formatFileSize,
    debounce,
    isValidPDF,
    createDownloadLink,
    createProgressBar,
    optimizePDFForWeb
};
