// Lazy Tool Loader - Load tool scripts on demand
class LazyToolLoader {
    constructor() {
        this.loadedTools = new Set();
        this.loadingTools = new Map();
        
        // Map of tool names to their script files
        this.toolScripts = {
            'pdf-editor': 'tools/pdf-editor.js',
            'resume-builder': 'tools/resume-builder.js',
            'image-converter': 'tools/image-converter.js',
            'remove-background': 'tools/remove-background.js',
            'image-effects': 'tools/image-effects.js',
            'qr-generator': 'tools/qr-generator.js',
            'base64-converter': 'tools/base64-converter.js',
            'favicon-generator': 'tools/favicon-generator.js',
            'color-extractor': 'tools/color-extractor.js',
            'bulk-rename': 'tools/bulk-rename.js',
            'collage-maker': 'tools/collage-maker.js',
            'meme-generator': 'tools/meme-generator.js',
            'pdf-splitter': 'tools/pdf-splitter.js',
            'image-resizer': 'tools/image-resizer.js',
            'image-cropper': 'tools/image-cropper.js',
            'url-shortener': 'tools/url-shortener.js',
            'text-extractor': 'tools/text-extractor.js',
            'watermark-tool': 'tools/watermark-tool.js',
            'pdf-rotator': 'tools/pdf-rotator.js',
            'batch-converter': 'tools/batch-converter.js',
            'image-optimizer': 'tools/image-optimizer.js',
            'pdf-compressor': 'tools/pdf-compressor.js',
            'video-to-gif': 'tools/video-to-gif.js',
            'image-filters': 'tools/image-filters.js',
            'pdf-merger-advanced': 'tools/pdf-merger-advanced.js',
            'screenshot-tool': 'tools/screenshot-tool.js',
            'image-comparison': 'tools/image-comparison.js',
            'pdf-form-filler': 'tools/pdf-form-filler.js',
            'background-remover': 'tools/background-remover.js',
            'background-changer': 'tools/background-changer.js',
            'background-blur': 'tools/background-blur.js',
            'jpg-to-png': 'tools/jpg-to-png.js',
            'png-to-jpg': 'tools/png-to-jpg.js',
            'webp-to-jpg': 'tools/webp-to-jpg.js',
            'webp-to-png': 'tools/webp-to-png.js',
            'png-to-webp': 'tools/png-to-webp.js',
            'jpg-to-webp': 'tools/jpg-to-webp.js',
            'grayscale-converter': 'tools/grayscale-converter.js',
            'sepia-filter': 'tools/sepia-filter.js',
            'image-rotate': 'tools/image-rotate.js',
            'image-flip': 'tools/image-flip.js',
            'heic-to-jpg': 'tools/heic-to-jpg.js',
            'word-counter': 'tools/word-counter.js',
            'space-counter': 'tools/space-counter.js',
            'typing-speed-master': 'tools/typing-speed-master.js',
            'password-generator': 'tools/password-generator.js',
            'gif-maker': 'tools/gif-maker.js',
            'metadata-remover': 'tools/metadata-remover.js',
            'invoice-generator': 'tools/invoice-generator.js',
            'pdf-page-extractor': 'tools/pdf-page-extractor.js',
            'image-border': 'tools/image-border.js',
            'svg-to-png': 'tools/svg-to-png.js',
            'video-trimmer': 'tools/video-trimmer.js',
            'unit-converter': 'tools/unit-converter.js',
            'currency-converter': 'tools/currency-converter.js',
            'qr-code-generator': 'tools/qr-code-generator.js',
            'barcode-generator': 'tools/barcode-generator.js',
            'text-to-speech': 'tools/text-to-speech.js',
            'base64-tool': 'tools/base64-tool.js',
            'hash-generator': 'tools/hash-generator.js',
            'json-formatter': 'tools/json-formatter.js',
            'csv-to-json': 'tools/csv-to-json.js',
            'url-tool': 'tools/url-tool.js',
            'markdown-to-html': 'tools/markdown-to-html.js',
            'color-picker': 'tools/color-picker.js',
            'image-watermark': 'tools/image-watermark.js',
            'pdf-watermark': 'tools/pdf-watermark.js',
            'lorem-ipsum-generator': 'tools/lorem-ipsum-generator.js',
            'uuid-generator': 'tools/uuid-generator.js',
            'image-upscaler': 'tools/image-upscaler.js',
            'image-splitter': 'tools/image-splitter.js',
            'pdf-highlighter': 'tools/pdf-highlighter.js',
            'pdf-bookmarker': 'tools/pdf-bookmarker.js',
            'png-to-svg': 'tools/png-to-svg.js',
            'screen-recorder': 'tools/screen-recorder.js',
            'timestamp-converter': 'tools/timestamp-converter.js',
            'case-converter': 'tools/case-converter.js',
            'diff-checker': 'tools/diff-checker.js',
            'regex-tester': 'tools/regex-tester.js',
            'html-tool': 'tools/html-tool.js',
            'css-minifier': 'tools/css-minifier.js',
            'js-minifier': 'tools/js-minifier.js',
            'sql-formatter': 'tools/sql-formatter.js',
            'xml-formatter': 'tools/xml-formatter.js',
            'yaml-to-json': 'tools/yaml-to-json.js',
            'random-generator': 'tools/random-generator.js',
            'ip-lookup': 'tools/ip-lookup.js',
            'stopwatch-timer': 'tools/stopwatch-timer.js',
            'char-frequency': 'tools/char-frequency.js',
            'duplicate-remover': 'tools/duplicate-remover.js',
            'line-sorter': 'tools/line-sorter.js',
            'pdf-metadata': 'tools/pdf-metadata.js',
            'pdf-table-extractor': 'tools/pdf-table-extractor.js',
            'image-collage': 'tools/image-collage.js',
            'signature-generator': 'tools/signature-generator.js',
            'whiteboard': 'tools/whiteboard.js',
            'json-to-csv': 'tools/json-to-csv.js',
            'excel-to-json': 'tools/excel-to-json.js',
            'meta-tag-generator': 'tools/meta-tag-generator.js',
            'sitemap-generator': 'tools/sitemap-generator.js',
            'age-calculator': 'tools/age-calculator.js',
            'expense-tracker': 'tools/expense-tracker.js',
            'video-compressor': 'tools/video-compressor.js',
            'audio-extractor': 'tools/audio-extractor.js',
            'photo-retoucher': 'tools/photo-retoucher.js',
            'image-merger': 'tools/image-merger.js',
            'pdf-header-footer': 'tools/pdf-header-footer.js',
            'gradient-generator': 'tools/gradient-generator.js',
            'icon-generator': 'tools/icon-generator.js',
            'pomodoro-timer': 'tools/pomodoro-timer.js',
            'changelog-generator': 'tools/changelog-generator.js',
            'image-to-text': 'tools/image-to-text.js',
            'email-signature': 'tools/email-signature.js',
            'certificate-generator': 'tools/certificate-generator.js',
            'business-card': 'tools/business-card.js',
            'presentation-maker': 'tools/presentation-maker.js',
            'mood-board': 'tools/mood-board.js',
            'mind-map': 'tools/mind-map.js',
            'url-qr-code': 'tools/url-qr-code.js',
            'video-thumbnail': 'tools/video-thumbnail.js',
            'audio-visualizer': 'tools/audio-visualizer.js',
            'merge-pdf': 'tools/merge-pdf.js',
            'split-pdf': 'tools/split-pdf.js',
            'zip-files': 'tools/zip-files.js',
            'unzip-files': 'tools/unzip-files.js',
            'organize-pdf': 'tools/organize-pdf.js',
            'rotate-pdf': 'tools/rotate-pdf.js',
            'crop-pdf': 'tools/crop-pdf.js',
            'page-numbers': 'tools/page-numbers.js',
            'compress-pdf': 'tools/compress-pdf.js',
            'compress-image': 'tools/compress-image.js',
            'compress-video': 'tools/compress-video.js',
            'resize-pdf': 'tools/resize-pdf.js',
            'pdf-to-word': 'tools/pdf-to-word.js',
            'word-to-pdf': 'tools/word-to-pdf.js',
            'pdf-to-image': 'tools/pdf-to-image.js',
            'image-to-pdf': 'tools/image-to-pdf.js',
            'pdf-to-jpg': 'tools/pdf-to-jpg.js',
            'jpg-to-pdf': 'tools/jpg-to-pdf.js',
            'pdf-to-ppt': 'tools/pdf-to-ppt.js',
            'ppt-to-pdf': 'tools/ppt-to-pdf.js',
            'pdf-to-excel': 'tools/pdf-to-excel.js',
            'excel-to-pdf': 'tools/excel-to-pdf.js',
            'html-to-pdf': 'tools/html-to-pdf.js',
            'pdf-to-pdfa': 'tools/pdf-to-pdfa.js',
            'ocr-pdf': 'tools/ocr-pdf.js',
            'edit-pdf': 'tools/edit-pdf.js',
            'watermark': 'tools/watermark.js',
            'sign-pdf': 'tools/sign-pdf.js',
            'redact-pdf': 'tools/redact-pdf.js',
            'protect-pdf': 'tools/protect-pdf.js',
            'unlock-pdf': 'tools/unlock-pdf.js',
            'compare-pdf': 'tools/compare-pdf.js',
            'repair-pdf': 'tools/repair-pdf.js',
            'scan-pdf': 'tools/scan-pdf.js',
            'image-base64': 'tools/image-base64.js',
            'pdf-ocr': 'tools/pdf-ocr.js',
            'text-analyzer': 'tools/text-analyzer.js'
        };
    }

    async loadTool(toolName) {
        // Already loaded
        if (this.loadedTools.has(toolName)) {
            console.log(`Tool ${toolName} already loaded`);
            return true;
        }

        // Currently loading
        if (this.loadingTools.has(toolName)) {
            console.log(`Tool ${toolName} is currently loading, waiting...`);
            return await this.loadingTools.get(toolName);
        }

        // Get script path
        const scriptPath = this.toolScripts[toolName];
        if (!scriptPath) {
            console.warn(`No script mapping for tool: ${toolName}`);
            console.log('Available tools:', Object.keys(this.toolScripts));
            return false;
        }

        // Ensure absolute path
        const absoluteScriptPath = scriptPath.startsWith('/') ? scriptPath : '/' + scriptPath;

        console.log(`Starting to load tool: ${toolName} from ${absoluteScriptPath}`);

        // Create loading promise
        const loadPromise = new Promise((resolve, reject) => {
            const script = document.createElement('script');
            // Add cache busting
            script.src = absoluteScriptPath + '?v=' + Date.now();
            script.async = true;
            
            script.onload = () => {
                this.loadedTools.add(toolName);
                this.loadingTools.delete(toolName);
                console.log(`✓ Loaded tool: ${toolName}`);
                resolve(true);
            };
            
            script.onerror = (error) => {
                this.loadingTools.delete(toolName);
                console.error(`✗ Failed to load tool script: ${scriptPath}`);
                reject(new Error(`Failed to load ${toolName}`));
            };
            
            document.body.appendChild(script);
        });

        this.loadingTools.set(toolName, loadPromise);
        return await loadPromise;
    }

    isToolLoaded(toolName) {
        return this.loadedTools.has(toolName);
    }
}

// Create global instance
window.lazyLoader = new LazyToolLoader();
