// Global variables
let currentTool = null;
let uploadedFiles = [];
let processingQueue = [];

// API base URL
const API_BASE = '/api';

// Open tool modal
function openTool(toolName) {
    currentTool = toolName;
    const modal = document.getElementById('toolModal');
    const modalBody = document.getElementById('modalBody');
    
    // List of newer container-based tools
    const containerBasedTools = [
        'pdf-splitter', 'image-resizer', 'image-cropper', 'url-shortener', 
        'text-extractor', 'watermark-tool', 'pdf-rotator', 'batch-converter',
        'image-optimizer', 'pdf-compressor', 'video-to-gif', 'image-filters',
        'pdf-merger-advanced', 'screenshot-tool', 'image-comparison', 'pdf-form-filler',
        'background-remover', 'background-changer', 'background-blur'
    ];
    
    if (containerBasedTools.includes(toolName)) {
        // For newer tools: create tool-container and call specific initializer
        modalBody.innerHTML = '<div id="tool-container"></div>';
        modal.style.display = 'block';
        
        // Convert tool-name to InitFunctionName (e.g., pdf-splitter -> initializePDFSplitter)
        const initFunctionName = 'initialize' + toolName.split('-').map(word => 
            word.charAt(0).toUpperCase() + word.slice(1)
        ).join('');
        
        // Try to call the initialization function
        try {
            if (typeof window[initFunctionName] === 'function') {
                window[initFunctionName]();
            } else if (typeof eval(initFunctionName) === 'function') {
                eval(initFunctionName + '()');
            } else {
                console.error('Initialization function not found:', initFunctionName);
                modalBody.innerHTML = '<div class="error-message">Tool initialization failed. Please refresh the page.</div>';
            }
        } catch (error) {
            console.error('Error initializing tool:', error);
            modalBody.innerHTML = '<div class="error-message">Tool initialization error: ' + error.message + '</div>';
        }
    } else {
        // For older tools: use the original interface generation
        modalBody.innerHTML = generateToolInterface(toolName);
        modal.style.display = 'block';
        initializeTool(toolName);
    }
}

// Close modal
function closeModal() {
    const modal = document.getElementById('toolModal');
    modal.style.display = 'none';
    uploadedFiles = [];
    processingQueue = [];
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('toolModal');
    if (event.target === modal) {
        closeModal();
    }
}

// Generate tool interface based on tool type
function generateToolInterface(toolName) {
    const toolConfigs = {
        'merge-pdf': {
            title: 'Merge PDF Files',
            description: 'Combine multiple PDF files in the order you want',
            icon: 'fa-object-group',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: []
        },
        'split-pdf': {
            title: 'Split PDF',
            description: 'Separate pages into independent PDF files',
            icon: 'fa-scissors',
            acceptedFiles: '.pdf',
            multiple: false,
            settings: [
                { type: 'select', label: 'Split Mode', id: 'splitMode', options: ['Split all pages', 'Extract specific pages', 'Split by page range'] }
            ]
        },
        'compress-pdf': {
            title: 'Compress PDF',
            description: 'Reduce file size while maintaining quality',
            icon: 'fa-file-zipper',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: [
                { type: 'range', label: 'Quality Level', id: 'quality', min: 1, max: 100, value: 85 }
            ]
        },
        'compress-image': {
            title: 'Compress Image',
            description: 'Reduce image size without quality loss',
            icon: 'fa-image',
            acceptedFiles: 'image/*',
            multiple: true,
            settings: [
                { type: 'range', label: 'Quality (%)', id: 'quality', min: 1, max: 100, value: 85 },
                { type: 'number', label: 'Target Size (KB)', id: 'targetSize', value: '' }
            ]
        },
        'compress-video': {
            title: 'Compress Video',
            description: 'Reduce video file size efficiently',
            icon: 'fa-video',
            acceptedFiles: 'video/*',
            multiple: true,
            settings: [
                { type: 'select', label: 'Quality', id: 'quality', options: ['High', 'Medium', 'Low'] },
                { type: 'number', label: 'Target Size (MB)', id: 'targetSize', value: '' }
            ]
        },
        'pdf-to-word': {
            title: 'PDF to Word',
            description: 'Convert PDF to editable DOC/DOCX',
            icon: 'fa-file-word',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: []
        },
        'word-to-pdf': {
            title: 'Word to PDF',
            description: 'Convert DOC/DOCX to PDF',
            icon: 'fa-file-pdf',
            acceptedFiles: '.doc,.docx',
            multiple: true,
            settings: []
        },
        'pdf-to-jpg': {
            title: 'PDF to JPG',
            description: 'Convert PDF pages to JPG images',
            icon: 'fa-file-image',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: [
                { type: 'range', label: 'Image Quality', id: 'quality', min: 1, max: 100, value: 95 }
            ]
        },
        'jpg-to-pdf': {
            title: 'JPG to PDF',
            description: 'Convert images to PDF',
            icon: 'fa-images',
            acceptedFiles: 'image/*',
            multiple: true,
            settings: [
                { type: 'select', label: 'Page Size', id: 'pageSize', options: ['A4', 'Letter', 'Legal', 'A3'] },
                { type: 'select', label: 'Orientation', id: 'orientation', options: ['Portrait', 'Landscape'] }
            ]
        },
        'image-converter': {
            title: 'Image Format Converter',
            description: 'Convert images between any formats',
            icon: 'fa-palette',
            acceptedFiles: 'image/*',
            multiple: true,
            settings: [],
            customInterface: true
        },
        'remove-background': {
            title: 'Remove Background',
            description: 'Remove image background automatically',
            icon: 'fa-scissors',
            acceptedFiles: 'image/*',
            multiple: true,
            settings: [
                { type: 'select', label: 'Background', id: 'bgType', options: ['Transparent', 'White', 'Black', 'Custom Color'] },
                { type: 'color', label: 'Custom Color', id: 'bgColor', value: '#ffffff' }
            ]
        },
        'image-effects': {
            title: 'Image Effects',
            description: 'Apply cool effects to images',
            icon: 'fa-magic',
            acceptedFiles: 'image/*',
            multiple: true,
            settings: [
                { type: 'select', label: 'Effect', id: 'effect', options: ['Vintage', 'Black & White', 'Sepia', 'Brighten', 'Darken', 'Invert', 'Blur', 'Sharpen', 'Edge Detect', 'Emboss'] }
            ]
        },
        'edit-pdf': {
            title: 'Edit PDF Live',
            description: 'Add text, images, shapes - like Canva for PDFs',
            icon: 'fa-wand-magic-sparkles',
            acceptedFiles: '.pdf',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'resume-builder': {
            title: 'Resume Builder',
            description: 'Create professional resumes',
            icon: 'fa-file-user',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'organize-pdf': {
            title: 'Organize PDF',
            description: 'Sort, delete, or add pages',
            icon: 'fa-sort',
            acceptedFiles: '.pdf',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'rotate-pdf': {
            title: 'Rotate PDF',
            description: 'Rotate your PDF pages',
            icon: 'fa-rotate',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: [
                { type: 'select', label: 'Rotation', id: 'rotation', options: ['90° Clockwise', '180°', '90° Counter-clockwise'] }
            ]
        },
        'watermark': {
            title: 'Add Watermark',
            description: 'Stamp image or text over your PDF',
            icon: 'fa-stamp',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: [
                { type: 'text', label: 'Watermark Text', id: 'watermarkText', value: '' },
                { type: 'select', label: 'Position', id: 'position', options: ['Center', 'Top Left', 'Top Right', 'Bottom Left', 'Bottom Right'] },
                { type: 'range', label: 'Opacity (%)', id: 'opacity', min: 0, max: 100, value: 50 }
            ]
        },
        'protect-pdf': {
            title: 'Protect PDF',
            description: 'Add password protection',
            icon: 'fa-lock',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: [
                { type: 'password', label: 'Password', id: 'password', value: '' }
            ]
        },
        'unlock-pdf': {
            title: 'Unlock PDF',
            description: 'Remove password security',
            icon: 'fa-unlock',
            acceptedFiles: '.pdf',
            multiple: true,
            settings: [
                { type: 'password', label: 'Current Password', id: 'password', value: '' }
            ]
        },
        'qr-generator': {
            title: 'QR Code Generator',
            description: 'Create custom QR codes',
            icon: 'fa-qrcode',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'base64-converter': {
            title: 'Base64 Converter',
            description: 'Encode/decode images to Base64',
            icon: 'fa-lock',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'favicon-generator': {
            title: 'Favicon Generator',
            description: 'Generate multi-size favicons',
            icon: 'fa-globe',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'color-extractor': {
            title: 'Color Palette Extractor',
            description: 'Extract colors from images',
            icon: 'fa-palette',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'bulk-rename': {
            title: 'Bulk File Renamer',
            description: 'Rename multiple files at once',
            icon: 'fa-tags',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'collage-maker': {
            title: 'Collage Maker',
            description: 'Create photo collages',
            icon: 'fa-images',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'meme-generator': {
            title: 'Meme Generator',
            description: 'Create custom memes',
            icon: 'fa-face-laugh',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'pdf-splitter': {
            title: 'Advanced PDF Splitter',
            description: 'Split PDFs by pages or ranges',
            icon: 'fa-scissors',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'image-resizer': {
            title: 'Bulk Image Resizer',
            description: 'Resize multiple images',
            icon: 'fa-expand-arrows-alt',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'image-cropper': {
            title: 'Smart Image Cropper',
            description: 'Crop with preset ratios',
            icon: 'fa-crop',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'url-shortener': {
            title: 'URL Shortener',
            description: 'Create short links with QR codes',
            icon: 'fa-link',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'text-extractor': {
            title: 'Text Extractor (OCR)',
            description: 'Extract text from images',
            icon: 'fa-file-text',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'watermark-tool': {
            title: 'Watermark Tool',
            description: 'Add watermarks to images',
            icon: 'fa-droplet',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'pdf-rotator': {
            title: 'PDF Page Rotator',
            description: 'Rotate PDF pages',
            icon: 'fa-rotate',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'batch-converter': {
            title: 'Batch Image Converter',
            description: 'Convert images to any format',
            icon: 'fa-exchange-alt',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'image-optimizer': {
            title: 'Image Optimizer',
            description: 'Optimize images for web',
            icon: 'fa-bolt',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'pdf-compressor': {
            title: 'PDF Compressor',
            description: 'Reduce PDF file sizes',
            icon: 'fa-file-zipper',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'video-to-gif': {
            title: 'Video to GIF Converter',
            description: 'Convert videos to animated GIFs',
            icon: 'fa-film',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'image-filters': {
            title: 'Image Filters & Effects',
            description: 'Apply filters and color adjustments',
            icon: 'fa-magic',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'pdf-merger-advanced': {
            title: 'Advanced PDF Merger',
            description: 'Merge PDFs with page selection',
            icon: 'fa-object-group',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'screenshot-tool-advanced': {
            title: 'Website Screenshot',
            description: 'Capture full-page screenshots',
            icon: 'fa-desktop',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'image-comparison': {
            title: 'Image Comparison Tool',
            description: 'Before/After slider comparison',
            icon: 'fa-clone',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'pdf-form-filler': {
            title: 'PDF Form Filler',
            description: 'Fill PDF forms automatically',
            icon: 'fa-file-signature',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'background-remover': {
            title: 'Background Remover',
            description: 'Remove image backgrounds instantly',
            icon: 'fa-user-slash',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'background-changer': {
            title: 'Background Changer',
            description: 'Replace image backgrounds',
            icon: 'fa-palette',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        },
        'background-blur': {
            title: 'Background Blur',
            description: 'Blur image backgrounds (portrait mode)',
            icon: 'fa-adjust',
            acceptedFiles: '',
            multiple: false,
            settings: [],
            customInterface: true
        }
    };

    const config = toolConfigs[toolName] || {
        title: 'Tool',
        description: 'Process your files',
        icon: 'fa-file',
        acceptedFiles: '*',
        multiple: true,
        settings: []
    };

    // For custom interfaces (PDF editor, resume builder)
    if (config.customInterface) {
        return generateCustomInterface(toolName, config);
    }

    // Standard upload interface
    return `
        <div class="tool-interface">
            <div class="tool-header">
                <div class="tool-icon" style="margin: 0 auto 1rem;">
                    <i class="fas ${config.icon}"></i>
                </div>
                <h2>${config.title}</h2>
                <p>${config.description}</p>
            </div>

            ${config.settings.length > 0 ? generateSettings(config.settings) : ''}

            <div class="upload-area" id="uploadArea" onclick="document.getElementById('fileInput').click()">
                <div class="upload-icon">
                    <i class="fas fa-cloud-upload-alt"></i>
                </div>
                <h3>Drop ${config.multiple ? 'files' : 'file'} here or click to upload</h3>
                <p>Accepted: ${config.acceptedFiles || 'All files'}</p>
                <input type="file" id="fileInput" class="file-input" 
                    accept="${config.acceptedFiles}" 
                    ${config.multiple ? 'multiple' : ''}>
            </div>

            <div id="fileList" class="file-list hidden"></div>

            <div id="progressContainer" class="progress-container hidden">
                <div class="progress-bar">
                    <div class="progress-fill" id="progressFill"></div>
                </div>
                <p class="progress-text" id="progressText">Processing...</p>
            </div>

            <div id="actionButtons" class="flex gap-2 mt-3 hidden">
                <button class="btn-primary" onclick="processFiles()" style="flex: 1">Process Files</button>
                <button class="btn-secondary" onclick="clearFiles()" style="flex: 1">Clear</button>
            </div>
        </div>
    `;
}

// Generate settings panel
function generateSettings(settings) {
    let html = '<div class="settings-panel">';
    
    settings.forEach(setting => {
        html += `<div class="setting-item">`;
        html += `<label for="${setting.id}">${setting.label}</label>`;
        
        switch (setting.type) {
            case 'select':
                html += `<select id="${setting.id}">`;
                setting.options.forEach(option => {
                    html += `<option value="${option}">${option}</option>`;
                });
                html += `</select>`;
                break;
            case 'range':
                html += `<input type="range" id="${setting.id}" 
                    min="${setting.min}" max="${setting.max}" value="${setting.value}"
                    oninput="document.getElementById('${setting.id}Value').textContent = this.value">`;
                html += `<span id="${setting.id}Value">${setting.value}</span>`;
                break;
            case 'number':
                html += `<input type="number" id="${setting.id}" value="${setting.value}" placeholder="Leave empty for auto">`;
                break;
            case 'text':
                html += `<input type="text" id="${setting.id}" value="${setting.value}" placeholder="${setting.label}">`;
                break;
            case 'password':
                html += `<input type="password" id="${setting.id}" value="${setting.value}" placeholder="Enter password">`;
                break;
        }
        
        html += `</div>`;
    });
    
    html += '</div>';
    return html;
}

// Generate custom interface for special tools
function generateCustomInterface(toolName, config) {
    if (toolName === 'edit-pdf') {
        // Initialize PDF editor after a short delay
        setTimeout(() => {
            if (window.initializePDFEditor) {
                window.initializePDFEditor();
            }
        }, 100);
        return `<div id="pdfEditor"></div>`;
    } else if (toolName === 'resume-builder') {
        // Initialize resume builder after a short delay
        setTimeout(() => {
            if (window.initializeResumeBuilder) {
                window.initializeResumeBuilder();
            }
        }, 100);
        return `<div id="resumeBuilder"></div>`;
    } else if (toolName === 'image-converter') {
        setTimeout(() => {
            initializeImageConverter();
        }, 100);
        return `<div id="imageConverter"></div>`;
    } else if (toolName === 'remove-background') {
        setTimeout(() => {
            initializeBackgroundRemover();
        }, 100);
        return `<div id="backgroundRemover"></div>`;
    } else if (toolName === 'image-effects') {
        setTimeout(() => {
            initializeImageEffects();
        }, 100);
        return `<div id="imageEffects"></div>`;
    } else if (toolName === 'qr-generator') {
        setTimeout(() => {
            if (window.initializeQRGenerator) {
                window.initializeQRGenerator();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'base64-converter') {
        setTimeout(() => {
            if (window.initializeBase64Converter) {
                window.initializeBase64Converter();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'favicon-generator') {
        setTimeout(() => {
            if (window.initializeFaviconGenerator) {
                window.initializeFaviconGenerator();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'color-extractor') {
        setTimeout(() => {
            if (window.initializeColorExtractor) {
                window.initializeColorExtractor();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'bulk-rename') {
        setTimeout(() => {
            if (window.initializeBulkRename) {
                window.initializeBulkRename();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'collage-maker') {
        setTimeout(() => {
            if (window.initializeCollageMaker) {
                window.initializeCollageMaker();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'meme-generator') {
        setTimeout(() => {
            if (window.initializeMemeGenerator) {
                window.initializeMemeGenerator();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'pdf-splitter') {
        setTimeout(() => {
            if (window.initializePDFSplitter) {
                window.initializePDFSplitter();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'image-resizer') {
        setTimeout(() => {
            if (window.initializeImageResizer) {
                window.initializeImageResizer();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'image-cropper') {
        setTimeout(() => {
            if (window.initializeImageCropper) {
                window.initializeImageCropper();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'url-shortener') {
        setTimeout(() => {
            if (window.initializeURLShortener) {
                window.initializeURLShortener();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'text-extractor') {
        setTimeout(() => {
            if (window.initializeTextExtractor) {
                window.initializeTextExtractor();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'watermark-tool') {
        setTimeout(() => {
            if (window.initializeWatermarkTool) {
                window.initializeWatermarkTool();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'pdf-rotator') {
        setTimeout(() => {
            if (window.initializePDFRotator) {
                window.initializePDFRotator();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'batch-converter') {
        setTimeout(() => {
            if (window.initializeBatchConverter) {
                window.initializeBatchConverter();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'image-optimizer') {
        setTimeout(() => {
            if (window.initializeImageOptimizer) {
                window.initializeImageOptimizer();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'pdf-compressor') {
        setTimeout(() => {
            if (window.initializePDFCompressor) {
                window.initializePDFCompressor();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'video-to-gif') {
        setTimeout(() => {
            if (window.initializeVideoToGIF) {
                window.initializeVideoToGIF();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'image-filters') {
        setTimeout(() => {
            if (window.initializeImageFilters) {
                window.initializeImageFilters();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'pdf-merger-advanced') {
        setTimeout(() => {
            if (window.initializePDFMergerAdvanced) {
                window.initializePDFMergerAdvanced();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'screenshot-tool-advanced') {
        setTimeout(() => {
            if (window.initializeScreenshotTool) {
                window.initializeScreenshotTool();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'image-comparison') {
        setTimeout(() => {
            if (window.initializeImageComparison) {
                window.initializeImageComparison();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'pdf-form-filler') {
        setTimeout(() => {
            if (window.initializePDFFormFiller) {
                window.initializePDFFormFiller();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'background-remover') {
        setTimeout(() => {
            if (window.initializeBackgroundRemover) {
                window.initializeBackgroundRemover();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'background-changer') {
        setTimeout(() => {
            if (window.initializeBackgroundChanger) {
                window.initializeBackgroundChanger();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    } else if (toolName === 'background-blur') {
        setTimeout(() => {
            if (window.initializeBackgroundBlur) {
                window.initializeBackgroundBlur();
            }
        }, 100);
        return `<div id="tool-container"></div>`;
    }
    return '';
}

// Initialize tool-specific functionality
function initializeTool(toolName) {
    const fileInput = document.getElementById('fileInput');
    const uploadArea = document.getElementById('uploadArea');

    if (!fileInput || !uploadArea) return;

    // File input change handler
    fileInput.addEventListener('change', handleFileSelect);

    // Drag and drop handlers
    uploadArea.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadArea.classList.add('dragover');
    });

    uploadArea.addEventListener('dragleave', () => {
        uploadArea.classList.remove('dragover');
    });

    uploadArea.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
        handleFileSelect({ target: { files: e.dataTransfer.files } });
    });
}

// Handle file selection
function handleFileSelect(event) {
    const files = Array.from(event.target.files);
    uploadedFiles = uploadedFiles.concat(files);
    displayFileList();
    
    document.getElementById('actionButtons').classList.remove('hidden');
}

// Display file list
function displayFileList() {
    const fileList = document.getElementById('fileList');
    if (!fileList) return;

    fileList.classList.remove('hidden');
    fileList.innerHTML = '';

    uploadedFiles.forEach((file, index) => {
        const fileItem = document.createElement('div');
        fileItem.className = 'file-item';
        fileItem.innerHTML = `
            <div class="file-info">
                <i class="fas ${getFileIcon(file.name)}"></i>
                <div class="file-details">
                    <h4>${file.name}</h4>
                    <p>${formatFileSize(file.size)}</p>
                </div>
            </div>
            <div class="file-actions">
                <button class="btn-icon" onclick="removeFile(${index})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        `;
        fileList.appendChild(fileItem);
    });
}

// Remove file from list
function removeFile(index) {
    uploadedFiles.splice(index, 1);
    displayFileList();
    
    if (uploadedFiles.length === 0) {
        document.getElementById('fileList').classList.add('hidden');
        document.getElementById('actionButtons').classList.add('hidden');
    }
}

// Clear all files
function clearFiles() {
    uploadedFiles = [];
    document.getElementById('fileList').classList.add('hidden');
    document.getElementById('actionButtons').classList.add('hidden');
    document.getElementById('fileInput').value = '';
}

// Process files
async function processFiles() {
    if (uploadedFiles.length === 0) {
        alert('Please select files first');
        return;
    }

    const progressContainer = document.getElementById('progressContainer');
    const progressFill = document.getElementById('progressFill');
    const progressText = document.getElementById('progressText');
    
    progressContainer.classList.remove('hidden');
    progressFill.style.width = '0%';

    const formData = new FormData();
    uploadedFiles.forEach(file => {
        formData.append('files', file);
    });

    // Add settings to form data
    const settings = getToolSettings();
    Object.keys(settings).forEach(key => {
        formData.append(key, settings[key]);
    });

    try {
        const endpoint = getToolEndpoint(currentTool);
        
        const xhr = new XMLHttpRequest();
        
        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                const percentComplete = (e.loaded / e.total) * 100;
                progressFill.style.width = percentComplete + '%';
                progressText.textContent = `Uploading... ${Math.round(percentComplete)}%`;
            }
        });

        xhr.addEventListener('load', () => {
            if (xhr.status === 200) {
                const response = JSON.parse(xhr.responseText);
                handleProcessingComplete(response);
            } else {
                throw new Error('Processing failed');
            }
        });

        xhr.addEventListener('error', () => {
            alert('An error occurred during processing');
            progressContainer.classList.add('hidden');
        });

        xhr.open('POST', endpoint);
        xhr.send(formData);

    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred: ' + error.message);
        progressContainer.classList.add('hidden');
    }
}

// Get tool settings
function getToolSettings() {
    const settings = {};
    const settingElements = document.querySelectorAll('.setting-item input, .setting-item select');
    
    settingElements.forEach(element => {
        if (element.id) {
            settings[element.id] = element.value;
        }
    });
    
    return settings;
}

// Get tool endpoint
function getToolEndpoint(toolName) {
    const endpoints = {
        'merge-pdf': `${API_BASE}/pdf/merge`,
        'split-pdf': `${API_BASE}/pdf/split`,
        'compress-pdf': `${API_BASE}/pdf/compress`,
        'compress-image': `${API_BASE}/image/compress`,
        'compress-video': `${API_BASE}/convert/compress-video`,
        'pdf-to-word': `${API_BASE}/convert/pdf-to-word`,
        'word-to-pdf': `${API_BASE}/convert/word-to-pdf`,
        'pdf-to-jpg': `${API_BASE}/convert/pdf-to-jpg`,
        'jpg-to-pdf': `${API_BASE}/convert/jpg-to-pdf`,
        'image-converter': `${API_BASE}/image/convert`,
        'rotate-pdf': `${API_BASE}/pdf/rotate`,
        'watermark': `${API_BASE}/pdf/watermark`,
        'protect-pdf': `${API_BASE}/pdf/protect`,
        'unlock-pdf': `${API_BASE}/pdf/unlock`
    };
    
    return endpoints[toolName] || `${API_BASE}/process`;
}

// Handle processing complete
function handleProcessingComplete(response) {
    const progressText = document.getElementById('progressText');
    const progressFill = document.getElementById('progressFill');
    
    progressFill.style.width = '100%';
    progressText.textContent = 'Complete! Downloading...';

    if (response.downloadUrl) {
        // Download single file
        downloadFile(response.downloadUrl, response.filename);
    } else if (response.files && response.files.length > 0) {
        // Download multiple files
        response.files.forEach(file => {
            downloadFile(file.url, file.name);
        });
    }

    setTimeout(() => {
        clearFiles();
        document.getElementById('progressContainer').classList.add('hidden');
        progressText.textContent = 'Processing...';
    }, 2000);
}

// Download file
function downloadFile(url, filename) {
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Utility functions
function getFileIcon(filename) {
    const ext = filename.split('.').pop().toLowerCase();
    const icons = {
        'pdf': 'fa-file-pdf',
        'doc': 'fa-file-word',
        'docx': 'fa-file-word',
        'jpg': 'fa-file-image',
        'jpeg': 'fa-file-image',
        'png': 'fa-file-image',
        'gif': 'fa-file-image',
        'mp4': 'fa-file-video',
        'avi': 'fa-file-video',
        'mov': 'fa-file-video'
    };
    return icons[ext] || 'fa-file';
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}
