// Simple SPA Router
class AppRouter {
    constructor(skipInitialRoute = false) {
        this.routes = {};
        this.currentRoute = null;
        this.skipInitialRoute = skipInitialRoute;
        this.init();
    }

    init() {
        // Handle navigation
        window.addEventListener('popstate', () => this.handleRoute());
        
        // Handle initial load (unless skipped)
        if (!this.skipInitialRoute) {
            this.handleRoute();
        }
        
        // Intercept link clicks
        document.addEventListener('click', (e) => {
            if (e.target.matches('[data-route]')) {
                e.preventDefault();
                const route = e.target.getAttribute('data-route');
                this.navigate(route);
            }
        });
    }

    register(path, handler) {
        this.routes[path] = handler;
    }

    navigate(path) {
        window.history.pushState({}, '', path);
        this.handleRoute();
    }

    handleRoute() {
        const path = window.location.pathname;
        console.log('=== ROUTE HANDLER ===');
        console.log('Current pathname:', path);
        this.currentRoute = path;

        // Check if it's a home route
        if (path === '/' || path === '/index.html') {
            console.log('Loading home');
            this.loadHome();
        } else {
            // Any other path is treated as a tool name
            // Remove leading slash to get tool name
            const toolName = path.replace(/^\/|\/$/g, '');
            
            // Ignore API routes and other system paths
            if (toolName && !toolName.startsWith('api/') && !toolName.startsWith('uploads/') && toolName !== 'favicon.ico') {
                console.log('Loading tool:', toolName);
                this.loadTool(toolName);
            } else {
                console.log('Loading home (system path)');
                this.loadHome();
            }
        }

        // Update active nav
        this.updateActiveNav(path);
    }

    loadHome() {
        const mainContent = document.getElementById('main-content');
        const toolPage = document.getElementById('tool-page');
        const hero = document.getElementById('main-hero');
        const categoryFilter = document.getElementById('category-filter');
        
        if (mainContent) {
            mainContent.style.display = 'block';
        }
        if (toolPage) {
            toolPage.classList.add('tool-page-hidden');
            toolPage.classList.remove('tool-page-visible');
        }
        if (hero) {
            hero.style.display = 'block';
        }
        if (categoryFilter) {
            categoryFilter.style.display = 'block';
        }
        
        console.log('Loaded HOME');
    }

    async loadTool(toolName) {
        console.log('=== LOAD TOOL ===');
        console.log('Loading tool:', toolName);
        
        const mainContent = document.getElementById('main-content');
        const toolPage = document.getElementById('tool-page');
        const toolContainer = document.getElementById('tool-container');
        const hero = document.getElementById('main-hero');
        const categoryFilter = document.getElementById('category-filter');
        
        console.log('Elements found:', {
            mainContent: !!mainContent,
            toolPage: !!toolPage,
            toolContainer: !!toolContainer,
            hero: !!hero,
            categoryFilter: !!categoryFilter
        });
        
        if (!mainContent || !toolPage || !toolContainer) {
            console.error('Required elements not found');
            return;
        }

        // Hide home, hero, and category filter; show tool page
        console.log('Setting display properties...');
        mainContent.style.display = 'none';
        toolPage.classList.remove('tool-page-hidden');
        toolPage.classList.add('tool-page-visible');
        if (hero) hero.style.display = 'none';
        if (categoryFilter) categoryFilter.style.display = 'none';
        console.log('Display properties set. toolPage classes:', toolPage.className);

        // Scroll to top
        window.scrollTo(0, 0);

        // Clear previous content with a minimal skeleton loader
        toolContainer.innerHTML = `
            <div class="tool-skeleton" style="max-width: 800px; margin: 0 auto; padding: 20px; opacity: 0.6; animation: pulse 1.5s infinite;">
                <div style="height: 40px; width: 60%; background: #eee; margin: 0 auto 20px; border-radius: 8px;"></div>
                <div style="height: 20px; width: 40%; background: #eee; margin: 0 auto 40px; border-radius: 4px;"></div>
                <div style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-bottom: 30px;">
                    <div style="height: 100px; background: #eee; border-radius: 10px;"></div>
                    <div style="height: 100px; background: #eee; border-radius: 10px;"></div>
                    <div style="height: 100px; background: #eee; border-radius: 10px;"></div>
                    <div style="height: 100px; background: #eee; border-radius: 10px;"></div>
                </div>
                <div style="height: 300px; background: #eee; border-radius: 10px;"></div>
            </div>
            <style>
                @keyframes pulse {
                    0% { opacity: 0.6; }
                    50% { opacity: 0.4; }
                    100% { opacity: 0.6; }
                }
            </style>
        `;

        // Tool name mappings (HTML names -> actual tool names)
        const toolMappings = {
            'image-base64': 'base64-converter',
            'remove-background': 'background-remover',
            'edit-pdf': 'pdf-editor',
            'screenshot-tool-advanced': 'screenshot-tool'
        };

        // Get the actual tool name
        const actualToolName = toolMappings[toolName] || toolName;

        // Lazy load the tool script
        if (window.lazyLoader) {
            try {
                console.log('Loading tool:', actualToolName);
                const loaded = await window.lazyLoader.loadTool(actualToolName);
                if (!loaded) {
                    throw new Error('Failed to load tool script');
                }
                // Wait for script to fully execute and register functions
                // Poll for the function to exist
                const initFuncName = 'initialize' + actualToolName.split('-').map(word => 
                    word.charAt(0).toUpperCase() + word.slice(1)
                ).join('');
                
                let attempts = 0;
                while (typeof window[initFuncName] === 'undefined' && attempts < 50) {
                    await new Promise(resolve => setTimeout(resolve, 100));
                    attempts++;
                }
                console.log(`Waited for ${initFuncName}: ${attempts * 100}ms`);
                
                // Fallback for word-counter if generic name fails
                if (actualToolName === 'word-counter' && typeof window[initFuncName] === 'undefined') {
                    console.log('Trying fallback for word-counter...');
                    if (typeof window.initializeWordCounter === 'function') {
                        console.log('Found window.initializeWordCounter directly');
                        window[initFuncName] = window.initializeWordCounter;
                    }
                }

                // Fallback for space-counter
                if (actualToolName === 'space-counter' && typeof window[initFuncName] === 'undefined') {
                    console.log('Trying fallback for space-counter...');
                    if (typeof window.initializeSpaceCounter === 'function') {
                        console.log('Found window.initializeSpaceCounter directly');
                        window[initFuncName] = window.initializeSpaceCounter;
                    }
                }

                // Fallback for typing-speed-master
                if (actualToolName === 'typing-speed-master' && typeof window[initFuncName] === 'undefined') {
                    console.log('Trying fallback for typing-speed-master...');
                    if (typeof window.initializeTypingSpeedMaster === 'function') {
                        console.log('Found window.initializeTypingSpeedMaster directly');
                        window[initFuncName] = window.initializeTypingSpeedMaster;
                    }
                }
            } catch (error) {
                console.error('Error lazy loading tool:', error);
                toolContainer.innerHTML = `
                    <div class="error-container">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h2>Failed to Load Tool</h2>
                        <p>Could not load the script for: ${actualToolName}</p>
                    </div>
                `;
                return;
            }
        }

        // Load the tool - try multiple naming conventions
        const initFunctionName = 'initialize' + actualToolName.split('-').map(word => 
            word.charAt(0).toUpperCase() + word.slice(1)
        ).join('');

        console.log('Looking for function:', initFunctionName);
        console.log('Function exists?', typeof window[initFunctionName]);
        console.log('All window.initialize* functions:', Object.keys(window).filter(k => k.startsWith('initialize')));

        try {
            // Try window.initializeFunctionName first
            if (typeof window[initFunctionName] === 'function') {
                console.log('Calling:', initFunctionName);
                window[initFunctionName]();
            }
            // Try global function
            else if (typeof window[initFunctionName.charAt(0).toLowerCase() + initFunctionName.slice(1)] === 'function') {
                window[initFunctionName.charAt(0).toLowerCase() + initFunctionName.slice(1)]();
            }
            // Try eval for locally defined functions
            else {
                console.warn(`Function ${initFunctionName} not found, trying legacy tool...`);
                try {
                    eval(initFunctionName + '()');
                } catch (e) {
                    // If all fails, try legacy tool
                    console.warn('Eval failed, trying legacy tool loader');
                    this.loadLegacyTool(toolName);
                }
            }
        } catch (error) {
            console.error('Error loading tool:', error);
            console.error('Tool name:', toolName);
            console.error('Init function:', initFunctionName);
            toolContainer.innerHTML = `
                <div class="error-container">
                    <i class="fas fa-exclamation-triangle"></i>
                    <h2>Tool Not Found</h2>
                    <p>The requested tool could not be loaded.</p>
                    <p style="color: #666; font-size: 14px;">Looking for: ${initFunctionName}</p>
                </div>
            `;
        }
    }

    loadLegacyTool(toolName) {
        // For older tools that use the modal system
        const toolContainer = document.getElementById('tool-container');
        
        // Map of legacy tool names to their configurations
        const legacyToolConfigs = {
            'merge-pdf': { title: 'Merge PDF Files', endpoint: '/api/pdf/merge', accept: '.pdf', multiple: true },
            'split-pdf': { title: 'Split PDF', endpoint: '/api/pdf/split', accept: '.pdf', multiple: false },
            'organize-pdf': { title: 'Organize PDF', endpoint: '/api/pdf/organize', accept: '.pdf', multiple: false },
            'rotate-pdf': { title: 'Rotate PDF', endpoint: '/api/pdf/rotate', accept: '.pdf', multiple: false },
            'crop-pdf': { title: 'Crop PDF', endpoint: '/api/pdf/crop', accept: '.pdf', multiple: false },
            'page-numbers': { title: 'Add Page Numbers', endpoint: '/api/pdf/page-numbers', accept: '.pdf', multiple: false },
            'compress-pdf': { title: 'Compress PDF', endpoint: '/api/pdf/compress', accept: '.pdf', multiple: true },
            'compress-image': { title: 'Compress Image', endpoint: '/api/image/compress', accept: 'image/*', multiple: true },
            'compress-video': { title: 'Compress Video', endpoint: '/api/convert/compress-video', accept: 'video/*', multiple: true },
            'resize-pdf': { title: 'Resize PDF', endpoint: '/api/pdf/resize', accept: '.pdf', multiple: false },
            'pdf-to-word': { title: 'PDF to Word', endpoint: '/api/convert/pdf-to-word', accept: '.pdf', multiple: true },
            'word-to-pdf': { title: 'Word to PDF', endpoint: '/api/convert/word-to-pdf', accept: '.doc,.docx', multiple: true },
            'pdf-to-jpg': { title: 'PDF to JPG', endpoint: '/api/convert/pdf-to-jpg', accept: '.pdf', multiple: true },
            'jpg-to-pdf': { title: 'JPG to PDF', endpoint: '/api/convert/jpg-to-pdf', accept: 'image/*', multiple: true },
            'pdf-to-ppt': { title: 'PDF to PowerPoint', endpoint: '/api/convert/pdf-to-ppt', accept: '.pdf', multiple: true },
            'ppt-to-pdf': { title: 'PowerPoint to PDF', endpoint: '/api/convert/ppt-to-pdf', accept: '.ppt,.pptx', multiple: true },
            'pdf-to-excel': { title: 'PDF to Excel', endpoint: '/api/convert/pdf-to-excel', accept: '.pdf', multiple: true },
            'excel-to-pdf': { title: 'Excel to PDF', endpoint: '/api/convert/excel-to-pdf', accept: '.xls,.xlsx', multiple: true },
            'html-to-pdf': { title: 'HTML to PDF', endpoint: '/api/convert/html-to-pdf', accept: '.html', multiple: true },
            'pdf-to-pdfa': { title: 'PDF to PDF/A', endpoint: '/api/convert/pdf-to-pdfa', accept: '.pdf', multiple: true },
            'ocr-pdf': { title: 'OCR PDF', endpoint: '/api/pdf/ocr', accept: '.pdf', multiple: false },
            'edit-pdf': { title: 'Edit PDF', endpoint: '/api/pdf/edit', accept: '.pdf', multiple: false },
            'watermark': { title: 'Add Watermark', endpoint: '/api/pdf/watermark', accept: '.pdf', multiple: false },
            'sign-pdf': { title: 'Sign PDF', endpoint: '/api/pdf/sign', accept: '.pdf', multiple: false },
            'redact-pdf': { title: 'Redact PDF', endpoint: '/api/pdf/redact', accept: '.pdf', multiple: false },
            'protect-pdf': { title: 'Protect PDF', endpoint: '/api/pdf/protect', accept: '.pdf', multiple: false },
            'unlock-pdf': { title: 'Unlock PDF', endpoint: '/api/pdf/unlock', accept: '.pdf', multiple: false },
            'compare-pdf': { title: 'Compare PDFs', endpoint: '/api/pdf/compare', accept: '.pdf', multiple: true },
            'repair-pdf': { title: 'Repair PDF', endpoint: '/api/pdf/repair', accept: '.pdf', multiple: false },
            'scan-pdf': { title: 'Scan to PDF', endpoint: '/api/pdf/scan', accept: 'image/*', multiple: true },
            'pdf-ocr': { title: 'PDF OCR', endpoint: '/api/pdf/ocr', accept: '.pdf', multiple: false },
            'screenshot-tool-advanced': { title: 'Screenshot Tool', endpoint: '/api/utility/screenshot', accept: '', multiple: false }
        };

        const config = legacyToolConfigs[toolName];
        
        if (config) {
            this.renderLegacyToolInterface(toolName, config);
        } else {
            throw new Error('Tool not found');
        }
    }

    renderLegacyToolInterface(toolName, config) {
        const toolContainer = document.getElementById('tool-container');
        
        toolContainer.innerHTML = `
            <div class="legacy-tool-interface">
                <div class="tool-header">
                    <h2>${config.title}</h2>
                    <p>Upload your files to get started</p>
                </div>

                <div class="upload-area" id="upload-area">
                    <div class="upload-icon">
                        <i class="fas fa-cloud-upload-alt"></i>
                    </div>
                    <h3>Click to Upload or Drag & Drop</h3>
                    <p>Select ${config.multiple ? 'one or more files' : 'a file'} to process</p>
                    <input type="file" id="file-input" class="file-input" 
                           accept="${config.accept}" 
                           ${config.multiple ? 'multiple' : ''}>
                </div>

                <div id="file-list" class="file-list hidden"></div>

                <div id="action-buttons" class="action-buttons hidden">
                    <button class="btn-primary" onclick="processLegacyTool('${toolName}', '${config.endpoint}')">
                        Process Files
                    </button>
                    <button class="btn-secondary" onclick="clearLegacyFiles()">
                        Clear All
                    </button>
                </div>

                <div id="progress-container" class="progress-container hidden">
                    <div class="progress-bar">
                        <div id="progress-fill" class="progress-fill"></div>
                    </div>
                    <p id="progress-text">Processing...</p>
                </div>

                <div id="results-container" class="results-container hidden"></div>
            </div>
        `;

        // Initialize legacy tool handlers
        this.initLegacyToolHandlers();
    }

    initLegacyToolHandlers() {
        const uploadArea = document.getElementById('upload-area');
        const fileInput = document.getElementById('file-input');

        if (!uploadArea || !fileInput) return;

        // Click to upload
        uploadArea.addEventListener('click', () => fileInput.click());

        // File selection
        fileInput.addEventListener('change', (e) => {
            window.legacyToolFiles = Array.from(e.target.files);
            this.displayLegacyFiles();
        });

        // Drag and drop
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });

        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });

        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            window.legacyToolFiles = Array.from(e.dataTransfer.files);
            this.displayLegacyFiles();
        });
    }

    displayLegacyFiles() {
        const fileList = document.getElementById('file-list');
        const actionButtons = document.getElementById('action-buttons');

        if (!window.legacyToolFiles || window.legacyToolFiles.length === 0) {
            fileList.classList.add('hidden');
            actionButtons.classList.add('hidden');
            return;
        }

        fileList.classList.remove('hidden');
        actionButtons.classList.remove('hidden');

        fileList.innerHTML = window.legacyToolFiles.map((file, index) => `
            <div class="file-item">
                <div class="file-info">
                    <i class="fas fa-file"></i>
                    <div class="file-details">
                        <h4>${file.name}</h4>
                        <p>${(file.size / 1024 / 1024).toFixed(2)} MB</p>
                    </div>
                </div>
                <button class="btn-icon" onclick="removeLegacyFile(${index})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        `).join('');
    }

    updateActiveNav(path) {
        document.querySelectorAll('.nav a').forEach(link => {
            if (link.getAttribute('href') === path) {
                link.classList.add('active');
            } else {
                link.classList.remove('active');
            }
        });
    }

    goBack() {
        window.history.back();
    }
}

// Initialize router only when DOM is ready
let router;

function initializeRouter() {
    // CRITICAL FIX: Check if we're on a tool page BEFORE creating router
    const path = window.location.pathname;
    console.log('[ROUTER INIT] Current path:', path);
    console.log('[ROUTER INIT] DOM ready state:', document.readyState);
    
    // Wait a micro-moment to ensure DOM elements are truly ready
    setTimeout(() => {
        // Check if it's NOT home and NOT an API/system path
        const isHomePath = path === '/' || path === '/index.html';
        const isSystemPath = path.startsWith('/api/') || path.startsWith('/uploads/') || path === '/favicon.ico';
        const isToolPath = !isHomePath && !isSystemPath;
        
        if (isToolPath) {
            console.log('[ROUTER INIT] Tool page detected, hiding home content BEFORE router init');
            const toolName = path.replace(/^\/|\/$/g, '');
            
            // Immediately hide home content BEFORE router.handleRoute() runs
            const mainContent = document.getElementById('main-content');
            const hero = document.getElementById('main-hero');
            const categoryFilter = document.getElementById('category-filter');
            const toolPage = document.getElementById('tool-page');
            
            console.log('[ROUTER INIT] Elements check:', {
                mainContent: !!mainContent,
                hero: !!hero,
                categoryFilter: !!categoryFilter,
                toolPage: !!toolPage
            });
            
            if (mainContent) {
                mainContent.style.display = 'none';
                console.log('[ROUTER INIT] Hidden main-content');
            }
            if (hero) {
                hero.style.display = 'none';
                console.log('[ROUTER INIT] Hidden hero');
            }
            if (categoryFilter) {
                categoryFilter.style.display = 'none';
                console.log('[ROUTER INIT] Hidden category-filter');
            }
            if (toolPage) {
                toolPage.classList.remove('tool-page-hidden');
                toolPage.classList.add('tool-page-visible');
                console.log('[ROUTER INIT] Showed tool-page with classes');
            }
            
            // Create router WITH handleRoute
            router = new AppRouter(false);
            console.log('[ROUTER INIT] Router created for tool:', toolName);
        } else {
            // Home page or other route - just create router normally
            console.log('[ROUTER INIT] Home page or other route detected');
            router = new AppRouter(false);
        }
    }, 0);
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeRouter);
} else {
    // DOM is already loaded
    initializeRouter();
}

// Global legacy tool files storage
window.legacyToolFiles = [];

// Navigation helper function
function openToolPage(toolName) {
    router.navigate(`/${toolName}`);
}

// Legacy tool helper functions
function clearLegacyFiles() {
    window.legacyToolFiles = [];
    const fileInput = document.getElementById('file-input');
    if (fileInput) fileInput.value = '';
    document.getElementById('file-list').classList.add('hidden');
    document.getElementById('action-buttons').classList.add('hidden');
}

function removeLegacyFile(index) {
    window.legacyToolFiles.splice(index, 1);
    router.displayLegacyFiles();
}

async function processLegacyTool(toolName, endpoint) {
    if (!window.legacyToolFiles || window.legacyToolFiles.length === 0) {
        alert('Please select files first');
        return;
    }

    const progressContainer = document.getElementById('progress-container');
    const progressFill = document.getElementById('progress-fill');
    const progressText = document.getElementById('progress-text');
    const resultsContainer = document.getElementById('results-container');

    progressContainer.classList.remove('hidden');
    progressFill.style.width = '0%';

    const formData = new FormData();
    window.legacyToolFiles.forEach(file => {
        formData.append('files', file);
    });

    try {
        const xhr = new XMLHttpRequest();

        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                const percentComplete = (e.loaded / e.total) * 100;
                progressFill.style.width = percentComplete + '%';
                progressText.textContent = `Uploading... ${Math.round(percentComplete)}%`;
            }
        });

        xhr.addEventListener('load', () => {
            if (xhr.status === 200) {
                const response = JSON.parse(xhr.responseText);
                displayLegacyResults(response, resultsContainer);
                progressContainer.classList.add('hidden');
            } else {
                throw new Error('Processing failed');
            }
        });

        xhr.addEventListener('error', () => {
            alert('An error occurred during processing');
            progressContainer.classList.add('hidden');
        });

        xhr.open('POST', endpoint);
        xhr.send(formData);

    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred: ' + error.message);
        progressContainer.classList.add('hidden');
    }
}

function displayLegacyResults(response, container) {
    container.classList.remove('hidden');
    
    if (response.success && response.downloadUrl) {
        container.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>Processing Complete!</h3>
                <p>Your file is ready for download.</p>
                <a href="${response.downloadUrl}" download class="btn-primary">
                    <i class="fas fa-download"></i> Download File
                </a>
            </div>
        `;
    } else if (response.files && response.files.length > 0) {
        container.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>Processing Complete!</h3>
                <p>${response.files.length} file(s) ready for download.</p>
                <div class="download-list">
                    ${response.files.map(file => `
                        <a href="${file.url}" download class="download-item">
                            <i class="fas fa-file"></i> ${file.name}
                        </a>
                    `).join('')}
                </div>
            </div>
        `;
    } else {
        container.innerHTML = `
            <div class="error-message">
                <i class="fas fa-exclamation-triangle"></i>
                <h3>Processing Failed</h3>
                <p>${response.error || 'An unknown error occurred'}</p>
            </div>
        `;
    }
}

// Dropdown Data Structure
const dropdownData = {
    organize: {
        icon: 'fa-sitemap',
        columns: [
            {
                title: 'Organize PDF',
                titleIcon: 'fa-sitemap',
                items: [
                    { name: 'Merge PDF', icon: 'fa-object-group' },
                    { name: 'Split PDF', icon: 'fa-cut' },
                    { name: 'Create ZIP', icon: 'fa-file-archive' },
                    { name: 'Extract ZIP', icon: 'fa-file-archive' },
                    { name: 'Remove Pages', icon: 'fa-trash' },
                    { name: 'Extract Pages', icon: 'fa-file-export' }
                ]
            },
            {
                title: 'Manage Pages',
                titleIcon: 'fa-layer-group',
                items: [
                    { name: 'Reorder Pages', icon: 'fa-arrows' },
                    { name: 'PDF Watermark', icon: 'fa-stamp' },
                    { name: 'Add Bookmarks', icon: 'fa-bookmark' },
                    { name: 'Rotate Pages', icon: 'fa-redo' }
                ]
            }
        ]
    },
    optimize: {
        icon: 'fa-tachometer-alt',
        columns: [
            {
                title: 'Optimize PDF',
                titleIcon: 'fa-tachometer-alt',
                items: [
                    { name: 'Compress PDF', icon: 'fa-compress' },
                    { name: 'Reduce PDF Size', icon: 'fa-minimize' },
                    { name: 'Auto Enhance', icon: 'fa-sparkles' },
                    { name: 'Batch Compress', icon: 'fa-layer-group' }
                ]
            },
            {
                title: 'Optimize Images',
                titleIcon: 'fa-image',
                items: [
                    { name: 'Compress Image', icon: 'fa-image' },
                    { name: 'Image Optimization', icon: 'fa-wand-magic-sparkles' },
                    { name: 'Resize Image', icon: 'fa-expand' },
                    { name: 'Image Quality', icon: 'fa-sliders-h' }
                ]
            }
        ]
    },
    convert: {
        icon: 'fa-exchange-alt',
        columns: [
            {
                title: 'Convert to PDF',
                titleIcon: 'fa-file-pdf',
                items: [
                    { name: 'Image to PDF', icon: 'fa-image' },
                    { name: 'JPG to PDF', icon: 'fa-image' },
                    { name: 'WORD to PDF', icon: 'fa-file-word' },
                    { name: 'POWERPOINT to PDF', icon: 'fa-file-powerpoint' },
                    { name: 'EXCEL to PDF', icon: 'fa-file-excel' }
                ]
            },
            {
                title: 'Convert from PDF',
                titleIcon: 'fa-exchange-alt',
                items: [
                    { name: 'PDF to Image', icon: 'fa-image' },
                    { name: 'PDF to WORD', icon: 'fa-file-word' },
                    { name: 'PDF to POWERPOINT', icon: 'fa-file-powerpoint' },
                    { name: 'PDF to EXCEL', icon: 'fa-file-excel' }
                ]
            },
            {
                title: 'Image Formats',
                titleIcon: 'fa-cube',
                items: [
                    { name: 'JPG to PNG', icon: 'fa-exchange' },
                    { name: 'PNG to JPG', icon: 'fa-exchange' },
                    { name: 'WebP Converter', icon: 'fa-cube' },
                    { name: 'Format Converter', icon: 'fa-arrows-alt' }
                ]
            }
        ]
    },
    edit: {
        icon: 'fa-pen',
        columns: [
            {
                title: 'Edit PDF',
                titleIcon: 'fa-pen-fancy',
                items: [
                    { name: 'PDF Editor', icon: 'fa-pen-fancy' },
                    { name: 'Rotate PDF', icon: 'fa-redo' },
                    { name: 'Crop PDF', icon: 'fa-crop' },
                    { name: 'Add Page Numbers', icon: 'fa-list-ol' }
                ]
            },
            {
                title: 'Edit Images',
                titleIcon: 'fa-paintbrush',
                items: [
                    { name: 'Image Editor', icon: 'fa-paintbrush' },
                    { name: 'Crop Image', icon: 'fa-crop' },
                    { name: 'Rotate Image', icon: 'fa-redo' },
                    { name: 'Add Watermark', icon: 'fa-stamp' }
                ]
            },
            {
                title: 'Annotations',
                titleIcon: 'fa-marker',
                items: [
                    { name: 'Annotation Tool', icon: 'fa-marker' },
                    { name: 'Add Comments', icon: 'fa-comment' },
                    { name: 'Highlight Text', icon: 'fa-highlighter' },
                    { name: 'Add Shapes', icon: 'fa-shapes' }
                ]
            }
        ]
    },
    security: {
        icon: 'fa-sparkles',
        columns: [
            {
                title: 'PDF Security',
                titleIcon: 'fa-lock',
                items: [
                    { name: 'Unlock PDF', icon: 'fa-lock-open' },
                    { name: 'Protect PDF', icon: 'fa-lock' },
                    { name: 'Sign PDF', icon: 'fa-pen-nib' },
                    { name: 'Redact PDF', icon: 'fa-eraser' }
                ]
            },
            {
                title: 'Image Tools',
                titleIcon: 'fa-wand-magic-sparkles',
                items: [
                    { name: 'Remove Background', icon: 'fa-scissors' },
                    { name: 'Change Background', icon: 'fa-image' },
                    { name: 'Blur Background', icon: 'fa-droplet' },
                    { name: 'Blur Image', icon: 'fa-droplet' }
                ]
            },
            {
                title: 'Analysis Tools',
                titleIcon: 'fa-chart-bar',
                items: [
                    { name: 'QR Code Generator', icon: 'fa-qrcode' },
                    { name: 'Text Analyzer', icon: 'fa-chart-bar' },
                    { name: 'Word Counter', icon: 'fa-calculator' },
                    { name: 'Typing Speed Test', icon: 'fa-keyboard' }
                ]
            }
        ]
    }
};

// Dropdown Handler
document.addEventListener('DOMContentLoaded', () => {
    const navWrappers = document.querySelectorAll('.nav-item-wrapper');

    navWrappers.forEach(wrapper => {
        const dropdownMenu = wrapper.querySelector('.dropdown-menu');
        const section = wrapper.getAttribute('data-section');

        // Populate dropdown on first hover
        wrapper.addEventListener('mouseenter', () => {
            if (!dropdownMenu || !dropdownMenu.innerHTML.trim()) {
                const data = dropdownData[section];
                if (data) {
                    let html = '';
                    let columnCount = 0;
                    
                    // Check if this category has custom columns (like convert)
                    if (data.columns) {
                        columnCount = data.columns.length;
                        // Render columns with titles
                        data.columns.forEach(column => {
                            html += `
                                <div class="dropdown-section">
                                    <div class="dropdown-title">
                                        <i class="fas ${column.titleIcon}"></i>
                                        ${column.title}
                                    </div>
                                    <ul class="dropdown-links">
                                        ${column.items.map(item => {
                                            const toolPath = '/' + item.name.toLowerCase().replace(/\s+/g, '-');
                                            return `
                                                <li><a href="${toolPath}" data-route="${toolPath}">
                                                    <i class="fas ${item.icon}"></i>
                                                    ${item.name}
                                                </a></li>
                                            `;
                                        }).join('')}
                                    </ul>
                                </div>
                            `;
                        });
                    } else {
                        // Original logic: Split items into 3 columns
                        columnCount = 3;
                        const itemsPerColumn = Math.ceil(data.items.length / 3);
                        
                        for (let i = 0; i < 3; i++) {
                            const columnItems = data.items.slice(i * itemsPerColumn, (i + 1) * itemsPerColumn);
                            
                            html += `
                                <div class="dropdown-section">
                                    <ul class="dropdown-links">
                                        ${columnItems.map(item => {
                                            const toolPath = '/' + item.name.toLowerCase().replace(/\s+/g, '-');
                                            return `
                                                <li><a href="${toolPath}" data-route="${toolPath}">
                                                    <i class="fas ${item.icon}"></i>
                                                    ${item.name}
                                                </a></li>
                                            `;
                                        }).join('')}
                                    </ul>
                                </div>
                            `;
                        }
                    }
                    
                    // Add explore more button that spans all columns
                    html += `
                        <div class="dropdown-explore" style="grid-column: 1 / -1;">
                            <a href="/#${section}" class="explore-btn" data-route="/">
                                <i class="fas fa-arrow-right"></i>
                                Explore More Tools
                            </a>
                        </div>
                    `;
                    
                    if (dropdownMenu) {
                        dropdownMenu.innerHTML = html;
                        dropdownMenu.setAttribute('data-columns', columnCount);
                    }
                }
            }
        });
    });

    // Hamburger menu
    const hamburger = document.querySelector('.hamburger-menu');
    if (hamburger) {
        hamburger.addEventListener('click', () => {
            hamburger.classList.toggle('active');
        });
    }

    // Logo click - go home
    const logo = document.querySelector('.logo');
    if (logo) {
        logo.addEventListener('click', (e) => {
            e.preventDefault();
            window.location.href = '/';
        });
    }
});
